# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to discover
        # a missing predicate (e.g., lock-known, temperature-known, durability-known, etc.)
        # by performing certain actions and observing the environment.

        # For demonstration, we will attempt to explore the environment by:
        # 1. Moving to each location to identify objects and their properties.
        # 2. Attempting to pick objects to learn their weight/durability.
        # 3. Attempting to pull drawers to learn their lock state.

        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']

        # Example: Suppose positions contains keys like 'drawer1', 'object1', 'location1', etc.
        # We'll iterate over all objects and try to perform exploration actions.

        # --- Step 1: Move to all known locations to identify objects ---
        # Assume positions contains a mapping: object_name -> (x, y, z) or location_name -> (x, y, z)
        # For this example, we will use the keys as location names if they start with 'location' or 'drawer'
        location_names = [k for k in positions.keys() if k.startswith('location') or k.startswith('drawer')]
        object_names = [k for k in positions.keys() if not (k.startswith('location') or k.startswith('drawer'))]

        # For logging and feedback
        exploration_log = []

        for loc in location_names:
            try:
                print(f"[Exploration] Moving to location: {loc}")
                # Use the move skill if available; here, we assume move(env, task, target_pos, ...)
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=positions[loc],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_log.append(f"Moved to {loc}")
                if done:
                    print(f"[Exploration] Task ended after moving to {loc}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        # --- Step 2: Try to pick each object to learn its properties (weight/durability) ---
        for obj in object_names:
            # Find the nearest location for the object
            # For simplicity, assume the object is at a location with a similar name
            possible_locs = [loc for loc in location_names if obj in loc or loc in obj]
            if possible_locs:
                obj_loc = possible_locs[0]
            else:
                # Fallback: just use the first location
                obj_loc = location_names[0] if location_names else None
            if obj_loc is None:
                print(f"[Exploration] No location found for object {obj}")
                continue
            try:
                print(f"[Exploration] Attempting to pick object: {obj} at {obj_loc}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=positions[obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_log.append(f"Picked {obj} at {obj_loc}")
                if done:
                    print(f"[Exploration] Task ended after picking {obj}!")
                    return
                # Optionally, place the object back
                print(f"[Exploration] Placing object: {obj} at {obj_loc}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=positions[obj_loc],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_log.append(f"Placed {obj} at {obj_loc}")
                if done:
                    print(f"[Exploration] Task ended after placing {obj}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to pick/place {obj} at {obj_loc}: {e}")
                continue

        # --- Step 3: Try to pull drawers to learn lock state ---
        for loc in location_names:
            if 'drawer' in loc:
                try:
                    print(f"[Exploration] Attempting to pull drawer: {loc}")
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=positions[loc],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    exploration_log.append(f"Pulled {loc}")
                    if done:
                        print(f"[Exploration] Task ended after pulling {loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to pull {loc}: {e}")
                    continue

        # --- Step 4: Try to rotate gripper at each location (if relevant) ---
        for loc in location_names:
            try:
                print(f"[Exploration] Attempting to rotate at location: {loc}")
                # For demonstration, use a fixed target quaternion (e.g., 90 degrees about z)
                target_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                exploration_log.append(f"Rotated at {loc}")
                if done:
                    print(f"[Exploration] Task ended after rotating at {loc}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to rotate at {loc}: {e}")
                continue

        print("[Exploration] Completed exploration phase.")
        print("[Exploration] Log:")
        for entry in exploration_log:
            print("  -", entry)

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
