# run_skeleton_task.py (Completed with Exploration and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time
import traceback

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration and feedback handling.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Ensure "rubbish" is in the object list ===
        # If not, log a warning (since we can't modify the PDDL here, just check and warn)
        if 'rubbish' not in positions:
            print("[Warning] 'rubbish' object not found in object positions! Please ensure it is declared in the object list.")

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to determine which predicate is missing.
        # We'll attempt to use available skills to interact with objects and log what we can/cannot do.

        # For demonstration, let's try to pick up all objects and see if any predicate is missing.
        # We'll also check for timeout and log failures.

        # Safety/force calibration placeholder (since we can't implement real force calibration here)
        def safety_check():
            # In a real system, check for collisions, force limits, etc.
            # Here, just print a message.
            print("[Safety] (Placeholder) Safety check passed.")

        # Set a global timeout for the task
        TASK_TIMEOUT = 60.0  # seconds
        start_time = time.time()

        # Track which predicates we can/cannot achieve
        exploration_results = {}

        # Try to pick each object in the environment
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to pick object: {obj_name} at position {obj_pos}")
            try:
                safety_check()
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj_name}.")
                    exploration_results[obj_name] = 'picked'
                else:
                    print(f"[Exploration] Pick action for {obj_name} did not complete the task.")
                    exploration_results[obj_name] = 'pick_incomplete'
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj_name}: {e}")
                traceback.print_exc()
                exploration_results[obj_name] = f'pick_failed: {e}'

            # Check for timeout
            elapsed = time.time() - start_time
            if elapsed > TASK_TIMEOUT:
                print(f"[Timeout] Exploration phase exceeded {TASK_TIMEOUT} seconds. Aborting exploration.")
                break

        # Log exploration results
        print("[Exploration] Results summary:")
        for obj, result in exploration_results.items():
            print(f"  - {obj}: {result}")

        # === Main Task Plan Execution ===
        # Here, you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, let's assume the plan is to pick 'rubbish' and place it in a 'bin' if both exist.

        if 'rubbish' in positions and 'bin' in positions:
            print("[Task] Attempting to pick 'rubbish' and place in 'bin'.")
            try:
                safety_check()
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=positions['rubbish'],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Picked up 'rubbish'.")
                else:
                    print("[Task] Pick action did not complete the task.")

                # Place in bin
                safety_check()
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=positions['bin'],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Placed 'rubbish' in 'bin'. Task complete.")
                else:
                    print("[Task] Place action did not complete the task.")

            except Exception as e:
                print(f"[Task] Failed to pick and place 'rubbish': {e}")
                traceback.print_exc()
        else:
            print("[Task] Either 'rubbish' or 'bin' not found in object positions. Skipping main task.")

        # === Timeout Handling ===
        total_elapsed = time.time() - start_time
        if total_elapsed > TASK_TIMEOUT:
            print(f"[Timeout] Task not completed within {TASK_TIMEOUT} seconds.")
        else:
            print(f"[Info] Task completed in {total_elapsed:.2f} seconds.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
