# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is blocking planning.
        # We will attempt to use available skills to interact with objects and locations,
        # and observe if any state or effect is missing (e.g., after pick, pull, etc.).
        # This phase is for debugging and exploration.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Try to interact with all objects and locations to see if any action fails due to missing predicates
        # For demonstration, we will attempt to:
        # 1. Move to each object/location
        # 2. Pick each object (if possible)
        # 3. Pull (if possible)
        # 4. Place (if possible)
        # 5. Rotate (if possible)
        # and log any errors or unexpected results.

        # Get all object names and their positions
        object_names = list(positions.keys())
        print("[Exploration] Objects found:", object_names)

        # Try to move to each object's position
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {target_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")

        # Try to pick each object
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick {obj_name} at {target_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj_name}: {e}")

        # Try to pull each object (if pull is applicable)
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pull {obj_name} at {target_pos}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after pulling {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull of {obj_name}: {e}")

        # Try to place each object (if place is applicable)
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to place {obj_name} at {target_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place of {obj_name}: {e}")

        # Try to rotate (if rotate is applicable)
        # For demonstration, try rotating the gripper to 90 degrees if possible
        try:
            print("[Exploration] Attempting to rotate gripper to 90 degrees")
            obs, reward, done = rotate(
                env,
                task,
                from_angle=0.0,
                to_angle=90.0,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        print("[Exploration] Exploration phase completed. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from exploration and feedback,
        # update the domain/problem or inform the user to add the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()