# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration knowledge suggest that some predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known) may be missing in the domain/problem.
        # We will perform a sequence of available skills to explore the environment and infer which predicate is missing.

        # For demonstration, we will:
        # 1. Move to each object location (if possible)
        # 2. Attempt to pick/place/pull/rotate as appropriate
        # 3. Observe if any action fails or if the environment state does not update as expected
        # 4. Print out observations to help identify missing predicates

        # Get a list of all objects and their positions
        object_names = list(positions.keys())
        print("[Exploration] Objects in environment:", object_names)

        # Try to move to each object's position and perform available actions
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {obj_pos}")
            try:
                # Move gripper to the object's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly after move to {obj_name}.")
                    return

                # Try to pick the object
                print(f"[Exploration] Attempting to pick {obj_name}")
                try:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking {obj_name}.")
                        return
                except Exception as e:
                    print(f"[Exploration] Pick failed for {obj_name}: {e}")

                # Try to place the object back (if pick succeeded)
                print(f"[Exploration] Attempting to place {obj_name}")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}.")
                        return
                except Exception as e:
                    print(f"[Exploration] Place failed for {obj_name}: {e}")

                # Try to rotate (if applicable)
                print(f"[Exploration] Attempting to rotate at {obj_name}")
                try:
                    # For demonstration, try rotating from 0 to 90 degrees if possible
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name='gripper',  # Replace with actual gripper name if needed
                        from_angle='zero_deg',
                        to_angle='ninety_deg'
                    )
                    if done:
                        print(f"[Exploration] Task ended after rotate at {obj_name}.")
                        return
                except Exception as e:
                    print(f"[Exploration] Rotate failed for {obj_name}: {e}")

                # Try to pull (if applicable)
                print(f"[Exploration] Attempting to pull {obj_name}")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        obj_name=obj_name
                    )
                    if done:
                        print(f"[Exploration] Task ended after pull on {obj_name}.")
                        return
                except Exception as e:
                    print(f"[Exploration] Pull failed for {obj_name}: {e}")

            except Exception as e:
                print(f"[Exploration] Exception during exploration of {obj_name}: {e}")

            # After each object, print the current observation for analysis
            obs = task.get_observation()
            print(f"[Exploration] Observation after interacting with {obj_name}: {obs}")

        print("[Exploration] Completed exploration of all objects.")
        print("[Exploration] Please review the above logs to identify which predicate is missing (e.g., if an action always fails or a state never updates, the corresponding predicate may be missing in the domain/problem).")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update your domain/problem files accordingly and rerun the plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()