# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': {...}, 'gripper': {...}, ...}
        positions = get_object_positions()

        # === Exploration Phase: Check Drawer Lock State ===
        # Feedback: (not (is-locked drawer1))
        # We need to determine if drawer1 is locked before attempting to pull.
        # Use the available 'pull' skill to explore lock state.
        # If the drawer is locked, the pull will fail or not open the drawer.
        # If not locked, the pull will succeed.

        # Assumptions for object names and positions
        drawer_name = 'drawer1'
        gripper_name = 'gripper'
        # These keys must exist in positions; handle missing keys gracefully.
        try:
            drawer_pos = positions[drawer_name]['position']
            gripper_pos = positions[gripper_name]['position']
        except Exception as e:
            print(f"[Error] Missing object positions: {e}")
            shutdown_environment(env)
            return

        # Step 1: Move gripper to the drawer handle position (if needed)
        # Use 'move' skill if available, else directly approach with 'pick' or 'pull'
        # For this domain, we assume the gripper is already near the drawer handle.

        # Step 2: Exploration - Try to pull the drawer to check lock state
        print("[Exploration] Attempting to pull the drawer to check if it is locked...")
        try:
            # Parameters for pull: env, task, pull_distance, pull_axis, ...
            # Assume pulling along 'x' axis by 0.10 meters
            pull_distance = 0.10
            pull_axis = 'x'
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis=pull_axis,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Pull action executed. Checking if drawer is open...")
            # After pull, check if the drawer is open (simulate predicate (is-open drawer1))
            # This can be checked via observation or environment state if available
            # For demonstration, we print the result and proceed
            # In a real system, you would check obs or task state for drawer open status
        except Exception as e:
            print(f"[Exploration] Pull failed, possibly due to locked drawer: {e}")
            print("[Exploration] Drawer is likely locked. Exploration complete.")
            shutdown_environment(env)
            return

        # Step 3: If pull succeeded, drawer is not locked. Proceed with the rest of the plan.
        print("[Exploration] Drawer is not locked. Proceeding with main task plan.")

        # === Main Task Plan ===
        # Example: Pick an object from inside the drawer and place it somewhere
        # For demonstration, we use placeholder object and location names
        object_to_pick = 'object1'
        pick_location = drawer_name  # Assume object is inside the drawer
        place_location = 'target_location'  # Replace with actual location as needed

        # Check if object exists in positions
        if object_to_pick not in positions:
            print(f"[Task] Object '{object_to_pick}' not found in positions. Skipping pick/place.")
        else:
            # Step 4: Pick the object from the drawer
            print(f"[Task] Picking up '{object_to_pick}' from '{pick_location}'...")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    obj=object_to_pick,
                    loc=pick_location
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Pick failed: {e}")
                shutdown_environment(env)
                return

            # Step 5: Place the object at the target location
            print(f"[Task] Placing '{object_to_pick}' at '{place_location}'...")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    obj=object_to_pick,
                    loc=place_location
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Place failed: {e}")
                shutdown_environment(env)
                return

        print("[Task] Main plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()