# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate (is-open drawer0) ---
        # Feedback indicates that (is-open drawer0) is missing or needs to be checked.
        # We will attempt to open the drawer using the available skills and check its state.

        # Assumptions for object names and positions:
        # - 'drawer0' is the drawer to be opened.
        # - 'gripper' is the robot's end effector.
        # - positions contains keys for 'drawer0', 'drawer0_handle', and possibly gripper positions.

        # You may need to adjust these keys based on your environment's naming conventions.
        drawer_name = 'drawer0'
        gripper_name = 'gripper'
        drawer_handle_name = 'drawer0_handle'
        anchor_pos_name = 'drawer0_anchor'
        side_pos_name = 'drawer0_side'
        ninety_deg_angle_name = 'ninety_deg'
        zero_deg_angle_name = 'zero_deg'

        # Retrieve positions for the drawer and handle
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Error] '{drawer_name}' not found in object positions.")
            return

        # For the sake of this example, we will use the handle position as the target for the gripper
        if drawer_handle_name in positions:
            handle_pos = positions[drawer_handle_name]
        else:
            handle_pos = drawer_pos  # fallback if handle not separately specified

        # Step 1: Move gripper to the side position of the drawer (if available)
        if side_pos_name in positions:
            side_pos = positions[side_pos_name]
            print(f"[Exploration] Moving gripper to side position of {drawer_name}: {side_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(side_pos))
                if done:
                    print("[Exploration] Task ended during move to side position!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to side position: {e}")
                return
        else:
            print(f"[Warning] Side position '{side_pos_name}' not found, moving to handle position instead.")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(handle_pos))
                if done:
                    print("[Exploration] Task ended during move to handle position!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to handle position: {e}")
                return

        # Step 2: Rotate gripper to 90 degrees (if required)
        # We assume the rotate skill takes (env, task, from_angle, to_angle)
        # If the current angle is not known, we attempt to rotate anyway
        try:
            print("[Exploration] Rotating gripper to 90 degrees.")
            obs, reward, done = rotate(env, task, from_angle=zero_deg_angle_name, to_angle=ninety_deg_angle_name)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            # Continue even if rotation fails

        # Step 3: Move to anchor position (if available)
        if anchor_pos_name in positions:
            anchor_pos = positions[anchor_pos_name]
            print(f"[Exploration] Moving gripper to anchor position of {drawer_name}: {anchor_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
                if done:
                    print("[Exploration] Task ended during move to anchor position!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to anchor position: {e}")
                return
        else:
            print(f"[Warning] Anchor position '{anchor_pos_name}' not found, using handle position.")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(handle_pos))
                if done:
                    print("[Exploration] Task ended during move to handle position!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to handle position: {e}")
                return

        # Step 4: Attempt to pull the drawer open
        try:
            print(f"[Exploration] Attempting to pull {drawer_name} open.")
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            return

        # Step 5: Check if the drawer is open (predicate: is-open drawer0)
        # This is the missing predicate from feedback.
        # We can check the state via the environment or observation if available.
        # For demonstration, print a message (actual check may depend on your env API).
        print(f"[Exploration] Checking if '{drawer_name}' is open (predicate: is-open {drawer_name}).")
        # If the observation provides this info, print it:
        obs = task.get_observation()
        if hasattr(obs, 'drawer_states'):
            drawer_states = getattr(obs, 'drawer_states')
            is_open = drawer_states.get(drawer_name, False)
            print(f"[Result] Drawer '{drawer_name}' open state: {is_open}")
        else:
            print("[Result] Drawer open state not available in observation. Please verify manually.")

        print("[Exploration] Exploration phase complete. Missing predicate identified: (is-open drawer0)")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()