# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'tomato1': (x, y, z), 'table1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at tomato1 table1)
        # The code must explore to determine which predicate is missing for the object at the location.
        # We will attempt to pick up tomato1 from table1, and if an error occurs, we infer a missing predicate.

        # Define object and location names based on feedback and typical naming
        tomato_name = 'tomato1'
        table_name = 'table1'

        # Get positions for tomato1 and table1
        try:
            tomato_pos = positions[tomato_name]
        except KeyError:
            print(f"[Error] Object '{tomato_name}' not found in positions dictionary.")
            shutdown_environment(env)
            return

        try:
            table_pos = positions[table_name]
        except KeyError:
            print(f"[Error] Location '{table_name}' not found in positions dictionary.")
            shutdown_environment(env)
            return

        # === Exploration: Try to pick tomato1 from table1 ===
        print(f"[Exploration] Attempting to pick '{tomato_name}' from '{table_name}' at position {tomato_pos}.")

        # Try to pick the object; if it fails, we suspect a missing predicate (e.g., identified, temperature-known, etc.)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Task ended after pick attempt on '{tomato_name}'.")
                return
            print(f"[Exploration] Successfully picked '{tomato_name}'. No missing predicate for pick.")
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            print("[Exploration] Likely missing predicate required before pick (e.g., identified, temperature-known, etc.).")
            # Here, you would log or record the missing predicate for further planning.
            # For this code, we simply report and stop.
            shutdown_environment(env)
            return

        # === If pick succeeded, try to place the object back on the table ===
        print(f"[Exploration] Attempting to place '{tomato_name}' on '{table_name}' at position {table_pos}.")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Task ended after place attempt on '{tomato_name}'.")
                return
            print(f"[Exploration] Successfully placed '{tomato_name}' on '{table_name}'.")
        except Exception as e:
            print(f"[Exploration] Exception during place: {e}")
            print("[Exploration] Place failed. Check for additional missing predicates or state requirements.")
            shutdown_environment(env)
            return

        # === Exploration Complete ===
        print("[Exploration] Exploration phase complete. If no exceptions, no missing predicate for pick/place.")

        # === (Optional) Continue with further plan steps as needed ===
        # For this exploration, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()