# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'tomato': (x,y,z), 'plate': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check if the robot can reach and identify objects, and if the pick action provides enough information.
        # We'll attempt to pick up each object and log the result, checking for reachability and feedback.

        # For demonstration, let's assume we have 'tomato' and 'plate' in the environment.
        # You may need to adjust these keys based on your actual object naming.
        object_keys = [k for k in positions.keys() if 'tomato' in k or 'plate' in k]
        tomatoes = [k for k in positions.keys() if 'tomato' in k]
        plates = [k for k in positions.keys() if 'plate' in k]

        # Log all available objects and their positions
        print("[Exploration] Detected objects and positions:")
        for obj, pos in positions.items():
            print(f"    {obj}: {pos}")

        # Safety: Check if required objects exist
        if not tomatoes:
            print("[Exploration] No tomatoes found in the environment. Exiting.")
            return
        if not plates:
            print("[Exploration] No plate found in the environment. Exiting.")
            return

        # For each tomato, try to pick and place on the plate
        for tomato in tomatoes:
            tomato_pos = positions[tomato]
            plate_pos = positions[plates[0]]  # Assume single plate

            print(f"[Task] Attempting to pick {tomato} at {tomato_pos}")

            # === Reachability Check ===
            # Try to move gripper above the tomato before picking
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=tomato_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to pick {tomato}: {e}")
                continue

            if done:
                print(f"[Task] Task ended after picking {tomato}!")
                return

            # Log successful pick
            print(f"[Exploration] Successfully picked {tomato}.")

            # === Place on Plate ===
            print(f"[Task] Attempting to place {tomato} on plate at {plate_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=plate_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to place {tomato} on plate: {e}")
                continue

            if done:
                print(f"[Task] Task ended after placing {tomato} on plate!")
                return

            print(f"[Exploration] Successfully placed {tomato} on plate.")

        # === Exploration: Check for Unreachable or Already Placed Objects ===
        # Try to pick up the tomato again (should fail if already on plate)
        for tomato in tomatoes:
            tomato_pos = positions[tomato]
            print(f"[Exploration] Checking if {tomato} can be picked again (should not be possible if already on plate).")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=tomato_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                print(f"[Exploration] Unexpectedly succeeded in picking {tomato} again. Possible missing predicate in domain.")
            except Exception as e:
                print(f"[Exploration] As expected, could not pick {tomato} again: {e}")

        # === Logging Final State ===
        print("[Exploration] Final state observation:")
        final_obs = task.get_observation()
        print(final_obs)

        print("[Exploration] Exploration phase complete. If objects could be picked up again after being placed, a predicate like (at ?obj ?loc) or (on ?obj ?loc) may be missing or not updated correctly in the domain.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()