# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'tomato2': (x, y, z), 'table1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at tomato2 table1)
        # The missing predicate is likely (at tomato2 table1), i.e., the robot must ensure tomato2 is at table1.
        # Exploration knowledge suggests that after moving to a location, the robot can identify objects there.
        # We will simulate this by moving to table1 and checking for tomato2.

        # For this example, we assume the robot's gripper is the manipulator and can move to table1.
        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # We use 'move' to approach table1, then 'pick' tomato2 if needed, and 'place' it at table1.

        # --- Step 1: Move to table1 to identify objects there ---
        try:
            print("[Exploration] Moving to table1 to identify objects.")
            # If 'move' skill requires positions, use positions['table1']
            # If 'move' skill requires names, use 'table1'
            # Since the skill_code is not shown, we assume move(env, task, target_pos, ...) signature
            table1_pos = positions.get('table1', None)
            if table1_pos is None:
                raise KeyError("table1 position not found in object_positions.")
            obs, reward, done = move(
                env,
                task,
                target_pos=table1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to table1!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to table1: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Check if tomato2 is at table1 ---
        tomato2_pos = positions.get('tomato2', None)
        if tomato2_pos is None:
            print("[Exploration] tomato2 position not found in object_positions.")
            shutdown_environment(env)
            return

        # If tomato2 is already at table1 (within a small threshold), no need to pick/place
        dist = np.linalg.norm(np.array(tomato2_pos) - np.array(table1_pos))
        if dist < 0.05:
            print("[Exploration] tomato2 is already at table1. No action needed.")
        else:
            # --- Step 3: Move to tomato2 and pick it up ---
            try:
                print("[Exploration] Moving to tomato2 to pick it up.")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=tomato2_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during move to tomato2!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to tomato2: {e}")
                shutdown_environment(env)
                return

            try:
                print("[Exploration] Picking up tomato2.")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=tomato2_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during pick of tomato2!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick of tomato2: {e}")
                shutdown_environment(env)
                return

            # --- Step 4: Move to table1 and place tomato2 there ---
            try:
                print("[Exploration] Moving to table1 to place tomato2.")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=table1_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during move to table1 (for place)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to table1 (for place): {e}")
                shutdown_environment(env)
                return

            try:
                print("[Exploration] Placing tomato2 at table1.")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=table1_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after placing tomato2!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place of tomato2: {e}")
                shutdown_environment(env)
                return

        print("[Exploration] Exploration phase complete. (at tomato2 table1) predicate is now true.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()