# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We will attempt to use available skills to interact with objects and observe the environment,
        # looking for any unexpected failures or missing state changes.

        # For demonstration, we will:
        # 1. Move to each object position (if possible)
        # 2. Attempt to pick and place each object (if possible)
        # 3. Attempt to rotate and pull (if available) on drawers
        # 4. Log any errors or unexpected results

        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # We will use only these.

        # For this exploration, we assume positions dict contains all relevant objects and their positions.
        # We'll try to interact with each object.

        # Keep track of which predicates might be missing by observing failures.
        missing_predicates = set()
        exploration_log = []

        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to move to {obj_name} at {obj_pos}")
            try:
                # Move the gripper to the object's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly after move to {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")
                missing_predicates.add('move')
                exploration_log.append((obj_name, 'move', str(e)))
                continue

            # Try to pick the object
            print(f"[Exploration] Attempting to pick {obj_name}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly after pick {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during pick {obj_name}: {e}")
                missing_predicates.add('pick')
                exploration_log.append((obj_name, 'pick', str(e)))
                continue

            # Try to place the object back at its original position (or a default place position)
            print(f"[Exploration] Attempting to place {obj_name}")
            try:
                # For simplicity, place at the same position (could be a bin or table in real tasks)
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly after place {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during place {obj_name}: {e}")
                missing_predicates.add('place')
                exploration_log.append((obj_name, 'place', str(e)))
                continue

        # Try to interact with drawers if any are present
        for obj_name, obj_pos in positions.items():
            if 'drawer' in obj_name.lower():
                print(f"[Exploration] Attempting to rotate {obj_name}")
                try:
                    # For demonstration, try rotating from 0 to 90 degrees (angles may need to be mapped)
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name='gripper',  # Replace with actual gripper name if available
                        from_angle='zero_deg',
                        to_angle='ninety_deg'
                    )
                    if done:
                        print(f"[Exploration] Task ended unexpectedly after rotate {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during rotate {obj_name}: {e}")
                    missing_predicates.add('rotate')
                    exploration_log.append((obj_name, 'rotate', str(e)))
                    continue

                print(f"[Exploration] Attempting to pull {obj_name}")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        gripper_name='gripper',  # Replace with actual gripper name if available
                        drawer_name=obj_name
                    )
                    if done:
                        print(f"[Exploration] Task ended unexpectedly after pull {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during pull {obj_name}: {e}")
                    missing_predicates.add('pull')
                    exploration_log.append((obj_name, 'pull', str(e)))
                    continue

        # === Exploration Summary ===
        if missing_predicates:
            print("\n[Exploration] Potential missing predicates or preconditions detected for actions:")
            for pred in missing_predicates:
                print(f"  - {pred}")
            print("[Exploration] Detailed log:")
            for log_entry in exploration_log:
                print(f"  Object: {log_entry[0]}, Action: {log_entry[1]}, Error: {log_entry[2]}")
        else:
            print("[Exploration] No missing predicates detected during skill execution.")

        # === End of Exploration Phase ===
        # At this point, you can use the exploration results to update your PDDL or planning logic.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
