# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests a missing predicate is blocking planning.
        # We will attempt to use available skills to interact with objects and locations,
        # and observe if any action fails or if the environment provides clues about missing predicates.

        # For demonstration, we will:
        # 1. Move to each object/location.
        # 2. Attempt to pick/place/pull/rotate as appropriate.
        # 3. Log any failures or unexpected results.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Get all object names and their positions
        object_names = list(positions.keys())

        # Attempt to move to each object's position
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {target_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to move to {obj_name}: {e}")

            # Try to pick the object (if possible)
            print(f"[Exploration] Attempting to pick {obj_name}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj_name}: {e}")

            # Try to place the object back (if possible)
            print(f"[Exploration] Attempting to place {obj_name}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to place {obj_name}: {e}")

            # Try to rotate (if the object is a gripper or has an angle)
            print(f"[Exploration] Attempting to rotate at {obj_name}")
            try:
                # For demonstration, try rotating between two angles if possible
                # These angle names are domain-specific; adjust as needed
                from_angle = 'zero_deg'
                to_angle = 'ninety_deg'
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name=obj_name,
                    from_angle=from_angle,
                    to_angle=to_angle
                )
                if done:
                    print(f"[Exploration] Task ended after rotating {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to rotate {obj_name}: {e}")

            # Try to pull (if the object is a drawer or can be pulled)
            print(f"[Exploration] Attempting to pull {obj_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    drawer_name=obj_name
                )
                if done:
                    print(f"[Exploration] Task ended after pulling {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to pull {obj_name}: {e}")

        print("[Exploration] Completed exploration of all objects.")

        # === End of Exploration Phase ===
        # At this point, based on the logs and any failures, you should be able to deduce
        # which predicate is missing (e.g., 'identified', 'temperature-known', 'weight-known', etc.)
        # and update your domain/problem files accordingly for future planning.

        # If you have a specific oracle plan, you would execute it here step by step
        # using the available skills, similar to the above.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
