# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at gripper nowhere-pos)
        # The feedback suggests that the predicate (at gripper nowhere-pos) is missing or not handled.
        # We need to explore the environment to determine the state of the gripper and its position.

        print("[Exploration] Checking gripper position and state...")

        try:
            # Try to get the gripper's position from the positions dictionary
            gripper_pos = positions.get('gripper', None)
            if gripper_pos is not None:
                print(f"[Exploration] Gripper position found: {gripper_pos}")
            else:
                print("[Exploration] Gripper position not found in positions dictionary.")
        except Exception as e:
            print(f"[Exploration] Exception while retrieving gripper position: {e}")

        # Check if the gripper is at 'nowhere-pos' (i.e., not at any known position)
        nowhere_pos = positions.get('nowhere-pos', None)
        if nowhere_pos is not None:
            print(f"[Exploration] nowhere-pos found: {nowhere_pos}")
        else:
            print("[Exploration] nowhere-pos not found in positions dictionary.")

        # If the predicate (at gripper nowhere-pos) is missing, we can log this for further debugging
        # and continue with the rest of the plan.

        # === Example Plan Execution ===
        # For demonstration, let's assume we want to pick an object and place it somewhere.
        # We'll use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # Find an object to pick (excluding 'gripper' and 'nowhere-pos')
        pick_obj = None
        pick_obj_pos = None
        for obj_name, obj_pos in positions.items():
            if obj_name not in ['gripper', 'nowhere-pos']:
                pick_obj = obj_name
                pick_obj_pos = obj_pos
                break

        if pick_obj is not None and pick_obj_pos is not None:
            print(f"[Task] Attempting to pick object '{pick_obj}' at position {pick_obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(pick_obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Task ended after picking '{pick_obj}'!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print("[Task] No suitable object found to pick.")

        # Optionally, place the object at a target location (if available)
        place_target = None
        for obj_name, obj_pos in positions.items():
            if obj_name not in ['gripper', 'nowhere-pos', pick_obj]:
                place_target = obj_pos
                break

        if place_target is not None:
            print(f"[Task] Attempting to place object '{pick_obj}' at position {place_target}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(place_target),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Task ended after placing '{pick_obj}'!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print("[Task] No suitable place target found.")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
