import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_robot_disposal_drawer_task():
    """Open a drawer, then pick the rubbish and drop it in the bin."""

    print("\n================  START TASK  ================\n")

    # ------------------------------------------------------------------
    #  Environment / task initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # reset task
        descriptions, obs = task.reset()

        # optional video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # obtain all object positions in a dict
        # keys must at least contain those in the supplied object list
        positions = get_object_positions()

        # ------------------------------------------------------------------
        #  Convenience variables
        # ------------------------------------------------------------------
        # choose the drawer we will operate on – here the “middle” one
        side_key = "middle_side_pos"
        anchor_key = "middle_anchor_pos"
        drawer_side_pos = np.array(positions[side_key])
        drawer_anchor_pos = np.array(positions[anchor_key])

        # rubbish item we need to throw away
        rubbish_key = "item3"
        rubbish_pos = np.array(positions[rubbish_key])

        # bin position
        bin_key = "bin"
        bin_pos = np.array(positions[bin_key])

        # small z‑offsets so that we approach from above
        TABLE_APPROACH_DZ = 0.15
        BIN_APPROACH_DZ = 0.15

        # ------------------------------------------------------------------
        #  STEP 1 – rotate gripper to 90° about z
        # ------------------------------------------------------------------
        current_quat = obs.gripper_pose[3:7]
        rot90_z = R.from_euler("z", 90, degrees=True).as_quat()
        # The rotate skill expects an absolute target quaternion.  Compose the
        # current orientation with the additional 90° rotation around z.
        # Quaternion multiplication (current * rot_add).
        current_R = R.from_quat(current_quat)
        target_quat = (current_R * R.from_quat(rot90_z)).as_quat()
        print("[Plan] STEP 1 – rotate gripper 90° about z")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Early Exit] task finished unexpectedly after rotate.")
            return

        # ------------------------------------------------------------------
        #  STEP 2 – move to drawer side position
        # ------------------------------------------------------------------
        print("[Plan] STEP 2 – move to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Early Exit] task finished unexpectedly after move‑to‑side.")
            return

        # ------------------------------------------------------------------
        #  STEP 3 – move to drawer anchor (handle) position
        # ------------------------------------------------------------------
        print("[Plan] STEP 3 – move to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Early Exit] task finished unexpectedly after move‑to‑anchor.")
            return

        # ------------------------------------------------------------------
        #  STEP 4 – grasp the drawer handle (pick‑drawer)
        #         (reuse generic pick skill – approach from +z)
        # ------------------------------------------------------------------
        print("[Plan] STEP 4 – grasp the drawer handle (pick‑drawer proxy)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.12,
            approach_axis="z",
        )
        if done:
            print("[Early Exit] task finished unexpectedly after picking handle.")
            return

        # ------------------------------------------------------------------
        #  STEP 5 – pull drawer open along +x (heuristic distance 0.18 m)
        # ------------------------------------------------------------------
        print("[Plan] STEP 5 – pull the drawer open (+x)")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.18,
            pull_axis="x",
        )
        if done:
            print("[Early Exit] task finished unexpectedly after pull.")
            return

        # release the handle (open gripper) so that we can grasp rubbish
        # easiest way: brief place at current position to open fingers
        action_pos = obs.gripper_pose[:3]
        place(
            env,
            task,
            target_pos=action_pos,
            approach_distance=0.00,
            approach_axis="z",
        )

        # ------------------------------------------------------------------
        #  STEP 6 – move above rubbish and pick it
        # ------------------------------------------------------------------
        rubbish_above = rubbish_pos + np.array([0.0, 0.0, TABLE_APPROACH_DZ])
        print("[Plan] STEP 6a – move above rubbish:", rubbish_above)
        move(env, task, target_pos=rubbish_above)

        print("[Plan] STEP 6b – pick rubbish at:", rubbish_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=TABLE_APPROACH_DZ,
            approach_axis="-z",
        )
        if done:
            print("[Early Exit] task finished unexpectedly after picking rubbish.")
            return

        # ------------------------------------------------------------------
        #  STEP 7 – move above bin and drop rubbish
        # ------------------------------------------------------------------
        bin_above = bin_pos + np.array([0.0, 0.0, BIN_APPROACH_DZ])
        print("[Plan] STEP 7a – move above bin:", bin_above)
        move(env, task, target_pos=bin_above)

        print("[Plan] STEP 7b – place rubbish into bin at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=BIN_APPROACH_DZ,
            approach_axis="-z",
        )

        if done:
            print("\n*****  TASK SUCCESS – Goal achieved!  *****")
        else:
            print("\n[Info] Task did not signal completion (done=False).")

    except Exception as e:
        print("[Exception] An error occurred:", e)
        raise

    finally:
        shutdown_environment(env)
        print("\n================  END TASK  ================\n")


if __name__ == "__main__":
    run_robot_disposal_drawer_task()
