import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from object_positions import get_object_positions
from video import init_video_writers, recording_step, recording_get_observation
from skill_code import rotate, move, pick, pull, place


def _safe_skill_call(skill_fn, *args, **kwargs):
    """Utility wrapper that executes a skill and gracefully handles common errors."""
    try:
        return skill_fn(*args, **kwargs)
    except Exception as e:
        print(f"[ERROR] Skill {skill_fn.__name__} failed with error: {e}")
        raise


def _quat_from_euler(x=0, y=0, z=0, degrees=True):
    """Returns (x, y, z, w) quaternion from Euler angles."""
    return R.from_euler('xyz', [x, y, z], degrees=degrees).as_quat()


def _get_pos(positions, key, fallback=(0.0, 0.0, 0.0)):
    """Fetch a position from the dictionary, warn if missing."""
    if key not in positions:
        print(f"[WARN] Position for '{key}' not found – using fallback {fallback}.")
    return np.array(positions.get(key, fallback), dtype=np.float32)


def run_drawer_and_cleanup_task():
    """
    Full task:
        1) Rotate gripper to correct orientation.
        2) Move to the chosen drawer side position.
        3) Move to the drawer anchor/handle.
        4) Grasp the drawer handle.
        5) Pull to open the drawer.
        6) Pick each rubbish item on the table.
        7) Place each rubbish item into the bin.
    """
    print("=========== [Task] START ===========")

    env, task = setup_environment()
    try:
        # Reset task and initialise video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all relevant positions
        positions = get_object_positions()

        # Drawer we will interact with (choose the 'bottom' drawer)
        drawer_side_key = 'bottom_side_pos'
        drawer_anchor_key = 'bottom_anchor_pos'

        drawer_side_pos = _get_pos(positions, drawer_side_key)
        drawer_anchor_pos = _get_pos(positions, drawer_anchor_key)

        # Rubbish items and bin
        rubbish_items = ['item1', 'item2', 'item3']
        bin_pos = _get_pos(positions, 'bin')

        # === Step 1 : rotate (specification) ===
        target_quat = _quat_from_euler(z=90)  # 90 deg around Z makes gripper sideways
        print("[Task] Step 1 – rotate to 90° about z‑axis.")
        obs, reward, done = _safe_skill_call(rotate, env, task, target_quat)
        if done:
            print("[Task] Task terminated unexpectedly during rotate.")
            return

        # === Step 2 : move to side position ===
        print(f"[Task] Step 2 – move to drawer side position '{drawer_side_key}'.")
        obs, reward, done = _safe_skill_call(move, env, task, drawer_side_pos)
        if done:
            print("[Task] Task terminated unexpectedly during move‑to‑side.")
            return

        # === Step 3 : move to anchor position ===
        print(f"[Task] Step 3 – move to drawer anchor/handle '{drawer_anchor_key}'.")
        obs, reward, done = _safe_skill_call(move, env, task, drawer_anchor_pos)
        if done:
            print("[Task] Task terminated unexpectedly during move‑to‑anchor.")
            return

        # === Step 4 : pick (grasp drawer handle) ===
        print("[Task] Step 4 – pick (close gripper on drawer handle).")
        obs, reward, done = _safe_skill_call(
            pick, env, task, drawer_anchor_pos, approach_axis='-z'
        )
        if done:
            print("[Task] Task terminated unexpectedly during drawer‑handle pick.")
            return

        # === Step 5 : pull to open drawer ===
        print("[Task] Step 5 – pull drawer open along +x direction.")
        obs, reward, done = _safe_skill_call(
            pull, env, task, pull_distance=0.15, pull_axis='x'
        )
        if done:
            print("[Task] Task terminated unexpectedly during pull.")
            return

        # === Steps 6 & 7 : pick each item and place into bin ===
        for item_name in rubbish_items:
            item_pos = _get_pos(positions, item_name)
            print(f"[Task] Pick‑and‑place for '{item_name}'.")

            # Step 6 : pick the item
            obs, reward, done = _safe_skill_call(
                pick, env, task, item_pos, approach_axis='-z'
            )
            if done:
                print(f"[Task] Task terminated unexpectedly while picking '{item_name}'.")
                return

            # Step 7 : place the item into the bin
            obs, reward, done = _safe_skill_call(
                place, env, task, bin_pos, approach_axis='-z'
            )
            if done:
                print(f"[Task] Task terminated unexpectedly while placing '{item_name}'.")
                return

        print("[Task] All rubbish successfully placed into bin. Task complete!")

    finally:
        shutdown_environment(env)
        print("=========== [Task] END ============")


if __name__ == "__main__":
    run_drawer_and_cleanup_task()