import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Oracle controller: open a drawer, then dispose of the rubbish in the bin."""
    print("===== Starting Open‑Drawer‑and‑Dispose Task =====")

    env, task = setup_environment()
    try:
        # --------------- Reset & Recording Wrappers -----------------
        descriptions, obs = task.reset()
        init_video_writers(obs)

        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------- Fetch Key Positions ------------------------
        positions = get_object_positions()

        # Drawer we will interact with (bottom preferred, else first found)
        drawer_side_key   = "bottom_side_pos"   if "bottom_side_pos"   in positions else \
                            next(k for k in positions if "side_pos"   in k)
        drawer_anchor_key = "bottom_anchor_pos" if "bottom_anchor_pos" in positions else \
                            next(k for k in positions if "anchor_pos" in k)

        side_pos   = np.array(positions[drawer_side_key])
        anchor_pos = np.array(positions[drawer_anchor_key])

        # Rubbish object (fallback to item3 or item1 if necessary)
        rubbish_key = "rubbish" if "rubbish" in positions else \
                      "item3"   if "item3"   in positions else "item1"
        rubbish_pos = np.array(positions[rubbish_key])

        bin_pos = np.array(positions["bin"])

        # --------------- Oracle Plan Execution ----------------------
        # STEP‑1: rotate gripper 90° about Z for drawer handle alignment
        current_quat = obs.gripper_pose[3:7]
        target_quat  = (R.from_euler("z", 90, degrees=True) *
                        R.from_quat(current_quat)).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Abort] Task ended during rotate.")
            return

        # STEP‑2: move to drawer side position
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Abort] Task ended during move to side.")
            return

        # STEP‑3: move to drawer anchor (handle) position
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Abort] Task ended during move to anchor.")
            return

        # STEP‑4: grasp the drawer handle
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Abort] Task ended during pick (drawer).")
            return

        # STEP‑5: pull the drawer open (20 cm along +X)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis="x")
        if done:
            print("[Abort] Task ended during pull.")
            return

        # STEP‑6: move above the rubbish lying on the table
        approach_rubbish = rubbish_pos + np.array([0.0, 0.0, 0.15])
        obs, reward, done = move(env, task, approach_rubbish)
        if done:
            print("[Abort] Task ended during move above rubbish.")
            return

        # STEP‑7: pick the rubbish
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("[Abort] Task ended during pick (rubbish).")
            return

        # STEP‑8: place the rubbish into the bin
        approach_bin = bin_pos + np.array([0.0, 0.0, 0.15])
        obs, reward, done = move(env, task, approach_bin)
        if done:
            print("[Abort] Task ended during move above bin.")
            return

        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Success] Goal achieved! Reward:", reward)
        else:
            print("[Info] Final place executed, but environment did not signal completion.")

    finally:
        shutdown_environment(env)

    print("===== End of Open‑Drawer‑and‑Dispose Task =====")


if __name__ == "__main__":
    run_skeleton_task()