import numpy as np
from scipy.spatial.transform import Rotation as R

# Environment / helper modules supplied by the framework
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# Pre‑implemented low‑level skills (do NOT redefine!)
from skill_code import move, rotate, pull, pick, place


def _choose_drawer(obj_pos):
    """
    Convenience helper:
    Pick the first drawer that provides both a side‑handle and an anchor position.
    Returns (side_pos, anchor_pos, drawer_prefix) if found, otherwise raises KeyError.
    """
    for prefix in ['bottom', 'middle', 'top']:
        side_key = f'{prefix}_side_pos'
        anchor_key = f'{prefix}_anchor_pos'
        if side_key in obj_pos and anchor_key in obj_pos:
            return obj_pos[side_key], obj_pos[anchor_key], prefix
    raise KeyError('[Planner] No drawer positions found in the scene!')


def _safe_offset(pos, dz=0.10):
    """
    Adds a vertical offset (along +Z) so that the gripper approaches from above.
    """
    return np.array(pos) + np.array([0.0, 0.0, dz])


def run_skeleton_task():
    print('=====  Task: Open Drawer & Throw Rubbish  =====')

    # ------------------------------------------------------------------
    # 1)  Environment initialisation / book‑keeping
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the task
        desc, obs = task.reset()

        # Optional video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)             # wrap .step
        task.get_observation = recording_get_observation( # wrap .get_observation
            task.get_observation
        )

        # ------------------------------------------------------------------
        # 2)  Retrieve all object poses needed for the plan
        # ------------------------------------------------------------------
        obj_pos = get_object_positions()

        # Drawer handle positions ------------------------------------------------
        side_pos, anchor_pos, drawer_prefix = _choose_drawer(obj_pos)
        print(f'[Planner] Using drawer: {drawer_prefix}')

        # Rubbish items on the table --------------------------------------------
        rubbish_items = [itm for itm in ['item1', 'item2', 'item3']
                         if itm in obj_pos]
        if not rubbish_items:
            print('[Planner] No rubbish found – nothing to do!')
            return

        # Bin position -----------------------------------------------------------
        if 'bin' not in obj_pos:
            raise KeyError('[Planner] Bin position not provided!')
        bin_drop_pos = _safe_offset(obj_pos['bin'], dz=0.15)  # a bit above the bin

        # ------------------------------------------------------------------
        # 3)  Follow the high‑level plan (matches specification steps)
        # ------------------------------------------------------------------
        # Step 1 - move: approach drawer side handle
        print('\n--- Step 1 / 8  :  Move to drawer side handle ---')
        obs, reward, done = move(env, task, _safe_offset(side_pos, dz=0.03))
        if done:
            print('[Task] Terminated unexpectedly at step 1.')
            return

        # Step 2 - rotate: turn gripper to 90° so it is perpendicular to handle
        print('\n--- Step 2 / 8  :  Rotate gripper 90° ---')
        target_quat = R.from_euler('xyz', [0, 0, np.deg2rad(90)]).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print('[Task] Terminated unexpectedly at step 2.')
            return

        # Step 3 - move: slide in towards anchor (handle grasp position)
        print('\n--- Step 3 / 8  :  Move to drawer anchor (handle) ---')
        obs, reward, done = move(env, task, _safe_offset(anchor_pos, dz=0.03))
        if done:
            print('[Task] Terminated unexpectedly at step 3.')
            return

        # Step 4 - pull: pull the drawer outward along +X
        print('\n--- Step 4 / 8  :  Pull drawer outward ---')
        # Positive X in RLBench usually opens drawers; distance tuned conservatively
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print('[Task] Terminated unexpectedly at step 4.')
            return

        # ------------------------------------------------------------------
        # 4)  For each rubbish item: pick from table, place into bin
        #     (Steps 5‑8 will be executed in this loop for every item)
        # ------------------------------------------------------------------
        for idx, item_name in enumerate(rubbish_items, start=1):
            print(f'\n=====  Disposing {item_name} ({idx}/{len(rubbish_items)})  =====')

            # Step 5 - move: hover above the rubbish item
            print('--- Step 5 / 8  :  Move above item on table ---')
            target_item_hover = _safe_offset(obj_pos[item_name], dz=0.10)
            obs, reward, done = move(env, task, target_item_hover)
            if done:
                print('[Task] Terminated unexpectedly while moving to item.')
                return

            # Step 6 - pick: grasp the rubbish item
            print('--- Step 6 / 8  :  Pick item ---')
            obs, reward, done = pick(env, task,
                                     target_pos=obj_pos[item_name],
                                     approach_distance=0.10,
                                     approach_axis='-z')
            if done:
                print('[Task] Terminated unexpectedly while picking.')
                return

            # Step 7 - move: hover above the bin
            print('--- Step 7 / 8  :  Move to bin ---')
            obs, reward, done = move(env, task, _safe_offset(bin_drop_pos, dz=0.0))
            if done:
                print('[Task] Terminated unexpectedly while moving to bin.')
                return

            # Step 8 - place: release the rubbish into the bin
            print('--- Step 8 / 8  :  Place item into bin ---')
            obs, reward, done = place(env, task,
                                      target_pos=bin_drop_pos,
                                      approach_distance=0.10,
                                      approach_axis='-z')
            if done:
                # Some tasks end immediately after successful placement
                print('[Task] Environment reported completion.')
                break

        print('\n=====  Task Completed – All rubbish disposed!  =====')

    finally:
        # Always, always cleanly shut down the simulator!
        shutdown_environment(env)


if __name__ == '__main__':
    run_skeleton_task()