import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from object_positions import get_object_positions
from video import init_video_writers, recording_step, recording_get_observation

# === Pre‑implemented skills (imported) ===
from skill_code import rotate, move, pick, pull, place


def _safe_pos(name: str, positions: dict):
    """Utility to safely fetch a position from the object‑positions dict."""
    if name not in positions:
        raise KeyError(f"[run_task] Position for '{name}' not found in object_positions.")
    return np.array(positions[name], dtype=np.float32)


def run_task():
    """Run the oracle plan that (1) opens an unlocked drawer, then (2) throws rubbish into the bin."""
    print("===== Starting Combined‑Domain Task =====")

    # ------------------------------------------------------------------
    #  Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the task / simulation
        descriptions, obs = task.reset()

        # Optional video‑recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)              # wrap step
        task.get_observation = recording_get_observation(  # wrap get_observation
            task.get_observation
        )

        # ------------------------------------------------------------------
        #  Retrieve all useful positions
        # ------------------------------------------------------------------
        positions = get_object_positions()
        bottom_side_pos   = _safe_pos('bottom_side_pos',   positions)
        bottom_anchor_pos = _safe_pos('bottom_anchor_pos', positions)
        rubbish_pos       = _safe_pos('rubbish',           positions)
        bin_pos           = _safe_pos('bin',               positions)

        # ------------------------------------------------------------------
        #  STEP‑1 : rotate gripper from zero_deg ➜ ninety_deg (around z‑axis)
        # ------------------------------------------------------------------
        print("\n[Step‑1] rotate gripper 0° ➜ 90° about Z")
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()   # xyzw
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Early‑Exit] Simulation ended during rotate.")
            return

        # ------------------------------------------------------------------
        #  STEP‑2 : move to drawer‑side position
        # ------------------------------------------------------------------
        print("\n[Step‑2] move gripper ➜ bottom_side_pos :", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Early‑Exit] Simulation ended during move‑to‑side.")
            return

        # ------------------------------------------------------------------
        #  STEP‑3 : move to drawer‑anchor (handle) position
        # ------------------------------------------------------------------
        print("\n[Step‑3] move gripper ➜ bottom_anchor_pos :", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Early‑Exit] Simulation ended during move‑to‑anchor.")
            return

        # ------------------------------------------------------------------
        #  STEP‑4 : pick the drawer handle (close gripper)
        # ------------------------------------------------------------------
        print("\n[Step‑4] pick (grasp) drawer handle at anchor position")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos,
                                 approach_distance=0.05,  # small approach since we are already near
                                 approach_axis='z')
        if done:
            print("[Early‑Exit] Simulation ended during drawer‑pick.")
            return

        # ------------------------------------------------------------------
        #  STEP‑5 : pull the drawer open
        #          – assume pulling along +x for 0.20 m
        # ------------------------------------------------------------------
        print("\n[Step‑5] pull the drawer open (+x, 0.20 m)")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Early‑Exit] Simulation ended during pull.")
            return

        # ------------------------------------------------------------------
        #  STEP‑6 : pick up the rubbish from the table
        # ------------------------------------------------------------------
        print("\n[Step‑6] pick rubbish on table :", rubbish_pos)
        obs, reward, done = pick(env, task, target_pos=rubbish_pos,
                                 approach_distance=0.15, approach_axis='z')
        if done:
            print("[Early‑Exit] Simulation ended during rubbish‑pick.")
            return

        # ------------------------------------------------------------------
        #  STEP‑7 : place rubbish into the bin
        # ------------------------------------------------------------------
        print("\n[Step‑7] place rubbish into bin :", bin_pos)
        obs, reward, done = place(env, task, target_pos=bin_pos,
                                  approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task Completed] Rubbish placed in bin. Reward:", reward)
        else:
            print("[Task] Finished action sequence but environment reports done=False.")

    finally:
        # Ensure simulator shuts down even on exception
        shutdown_environment(env)

    print("===== End of Combined‑Domain Task =====")


if __name__ == "__main__":
    run_task()