# run_open_drawer_and_dispose.py

import numpy as np
from scipy.spatial.transform import Rotation as R                   # <‑‑ needed for quaternion handling
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Skill imports (pre‑implemented primitives) ===
from skill_code import rotate, move, pick, pull, place             # do NOT redefine these!

# -------------------------------------------------------------------------------------
# Helper: produce a 90‑deg rotation quaternion around Z (x‑y‑z‑w convention → xyzw array)
def quat_z_90deg():
    return R.from_euler('z', 90, degrees=True).as_quat()

# -------------------------------------------------------------------------------------
def run_task_open_drawer_and_dispose():
    """
    1) Rotate gripper 90° (side‑grasp pose)
    2) Move to chosen drawer side position
    3) Move to drawer anchor position
    4) Grasp drawer handle (pick)
    5) Pull drawer out
    6) Pick the rubbish object
    7) Place rubbish into bin
    """

    print("\n================  START TASK  =================")

    # ------------------------------------------------------------------  Environment
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (optional) video
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------------------  Object information
        pos = get_object_positions()      # dictionary of positions for all named objects

        # Mandatory objects we expect to find
        REQUIRED_KEYS = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'rubbish', 'bin'
        ]
        for k in REQUIRED_KEYS:
            if k not in pos:
                raise RuntimeError(f"[Task] Required object position '{k}' not found!")

        side_pos         = np.asarray(pos['bottom_side_pos'])
        anchor_pos       = np.asarray(pos['bottom_anchor_pos'])
        rubbish_pos      = np.asarray(pos['rubbish'])
        bin_pos          = np.asarray(pos['bin'])

        # --------------------------------------------------------------  STEP‑BY‑STEP
        done = False
        reward = 0.0

        # Step‑1  ROTATE gripper to 90°
        if not done:
            target_quat = quat_z_90deg()
            print("\n[Step‑1] ROTATE gripper 90°")
            obs, reward, done = rotate(env, task, target_quat)

        # Step‑2  MOVE to drawer side position
        if not done:
            print("\n[Step‑2] MOVE to drawer SIDE position:", side_pos)
            obs, reward, done = move(env, task, target_pos=side_pos)

        # Step‑3  MOVE to drawer anchor (handle) position
        if not done:
            print("\n[Step‑3] MOVE to drawer ANCHOR position:", anchor_pos)
            obs, reward, done = move(env, task, target_pos=anchor_pos)

        # Step‑4  PICK the drawer handle
        if not done:
            print("\n[Step‑4] PICK drawer handle at anchor position")
            obs, reward, done = pick(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.05,
                approach_axis='-z'
            )

        # Step‑5  PULL the drawer out (open)
        if not done:
            print("\n[Step‑5] PULL drawer to open")
            obs, reward, done = pull(
                env, task,
                pull_distance=0.20,       # 20 cm pull
                pull_axis='x'             # assume drawer slides along +X
            )

        # Step‑6  (optional) MOVE to rubbish if far away
        #         We attempt a direct move; if too close already, move handles it quickly.
        if not done:
            print("\n[Step‑6a] MOVE to rubbish on table:", rubbish_pos)
            obs, reward, done = move(env, task, target_pos=rubbish_pos)

        # Step‑6b PICK rubbish
        if not done:
            print("\n[Step‑6b] PICK rubbish")
            obs, reward, done = pick(
                env, task,
                target_pos=rubbish_pos,
                approach_distance=0.10,
                approach_axis='-z'
            )

        # Step‑7  MOVE to bin
        if not done:
            print("\n[Step‑7a] MOVE to bin:", bin_pos)
            obs, reward, done = move(env, task, target_pos=bin_pos)

        # Step‑7b PLACE rubbish in bin
        if not done:
            print("\n[Step‑7b] PLACE rubbish in bin")
            obs, reward, done = place(
                env, task,
                target_pos=bin_pos,
                approach_distance=0.10,
                approach_axis='-z'
            )

        # ---------------------------------------------------------  Completion status
        if done:
            print("\n[Task] Task ended early (done=True).  Reward:", reward)
        else:
            print("\n[Task] Task completed sequence.  Final reward maybe:", reward)

    finally:
        shutdown_environment(env)
        print("================  END TASK  =================\n")


if __name__ == "__main__":
    run_task_open_drawer_and_dispose()
