# run_skeleton_task.py  – Executable task logic that follows the oracle plan
#
# The code below extends the generic skeleton so that the robot:
#   1. Rotates the gripper from 0° to 90°
#   2. Moves to the side position of the middle drawer
#   3. Moves to the anchor position of the middle drawer
#   4. Grasps the drawer handle
#   5. Pulls the drawer open
#   6. Picks the “rubbish” object from the table
#   7. Drops / places the rubbish in the bin
#
# Only the predefined skills (rotate, move, pick, pull, place) are used.

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull          # predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions               # helper for object poses


def safe_get_pos(name, positions):
    """Return the 3‑D position of an object or raise a clear error."""
    if name not in positions:
        raise KeyError(f"[run_task] Object ‘{name}’ not found in position dictionary.")
    return np.asarray(positions[name], dtype=np.float32)


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  1) Set‑up environment
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video/observation recording
        init_video_writers(obs)
        task.step = recording_step(task.step)                      # wrap step
        task.get_observation = recording_get_observation(          # wrap get_obs
            task.get_observation
        )

        # ------------------------------------------------------------------
        #  2) Fetch all object positions we need
        # ------------------------------------------------------------------
        positions = get_object_positions()

        middle_side_pos   = safe_get_pos("middle_side_pos",   positions)
        middle_anchor_pos = safe_get_pos("middle_anchor_pos", positions)
        rubbish_pos       = safe_get_pos("rubbish",           positions)
        bin_pos           = safe_get_pos("bin",               positions)

        # ------------------------------------------------------------------
        #  3) Execute oracle plan
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # --- Step 1: rotate(gripper, zero_deg, ninety_deg) ---------------
        # We interpret “ninety_deg” as a +90° rotation about the Z‑axis.
        target_quat_xyzw = R.from_euler("z", 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat_xyzw)
        if done:
            print("[Plan] Task ended during rotate!"); return

        # --- Step 2: move‑to‑side(gripper, middle, nowhere, side‑pos‑middle)
        print("[Plan] Moving to middle_side_pos:", middle_side_pos)
        obs, reward, done = move(env, task, target_pos=middle_side_pos)
        if done:
            print("[Plan] Task ended during move‑to‑side!"); return

        # --- Step 3: move‑to‑anchor(gripper, middle, side‑pos‑middle, anchor‑pos‑middle)
        print("[Plan] Moving to middle_anchor_pos:", middle_anchor_pos)
        obs, reward, done = move(env, task, target_pos=middle_anchor_pos)
        if done:
            print("[Plan] Task ended during move‑to‑anchor!"); return

        # --- Step 4: pick‑drawer(gripper, middle, anchor‑pos‑middle)
        print("[Plan] Picking the middle drawer handle.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=middle_anchor_pos,
            approach_distance=0.10,
            approach_axis="z"
        )
        if done:
            print("[Plan] Task ended during pick‑drawer!"); return

        # --- Step 5: pull(gripper, middle)  – pull the drawer open --------
        print("[Plan] Pulling the drawer open (~0.15 m along +x).")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,       # 15 cm pull
            pull_axis="x"
        )
        if done:
            print("[Plan] Task ended during pull!"); return

        # Optional: back up a little after opening drawer (quality‑of‑life)
        backup_pos = middle_anchor_pos + np.array([-0.05, 0.0, 0.0])  # 5 cm back
        print("[Plan] Backing away from drawer handle.")
        obs, reward, done = move(env, task, target_pos=backup_pos)
        if done:
            print("[Plan] Task ended while backing away!"); return

        # --- Step 6: pick(rubbish, table) ---------------------------------
        print("[Plan] Picking rubbish at:", rubbish_pos)
        obs, reward, done = move(env, task, target_pos=rubbish_pos + np.array([0, 0, 0.10]))
        if done: return
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.10,
            approach_axis="z"
        )
        if done:
            print("[Plan] Task ended during pick(rubbish)!"); return

        # --- Step 7: place(rubbish, bin) ----------------------------------
        print("[Plan] Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.12,
            approach_axis="z"
        )

        # ------------------------------------------------------------------
        #  4) Completion message
        # ------------------------------------------------------------------
        if done:
            print("[Plan] Task completed successfully! Reward:", reward)
        else:
            print("[Plan] Task finished all steps but ‘done’ flag is False – check success criteria.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()