# run_combined_drawer_disposal.py
#
# Description:
#   Executes the oracle plan specified in the specification section:
#       1) Rotate the gripper to ninety degrees.
#       2) Move next to the bottom drawer (side‑pos).
#       3) Move to the drawer’s anchor‑pos (handle).
#       4) Grasp the drawer handle.
#       5) Pull until the drawer is fully open.
#       6) Pick up the rubbish that is on the table.
#       7) Drop the rubbish into the bin.
#
#   Only the already‑provided primitive skills (rotate, move, pick,
#   pull, place) are used.

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)
from object_positions import get_object_positions


def _quat_from_euler(deg_x: float, deg_y: float, deg_z: float):
    """Utility: returns (x, y, z, w) quaternion from euler degrees."""
    return R.from_euler('xyz',
                        [np.deg2rad(deg_x),
                         np.deg2rad(deg_y),
                         np.deg2rad(deg_z)]).as_quat()


def _axis_and_distance(src: np.ndarray,
                       dst: np.ndarray,
                       default_distance: float = 0.20):
    """
    Decide which axis (x / y / z) and signed distance best matches the
    vector from src → dst so the pull skill can open the drawer.

    Returns:
        axis_str (e.g. 'x', '-y', ...) , distance (positive float)
    """
    vec = dst - src
    # If the supplied vector is too small (e.g. pos coincident) fall
    # back to the default pull along +x.
    if np.linalg.norm(vec) < 1e-3:
        return 'x', default_distance

    # Find dominant component (|x|,|y|,|z|)
    axis_idx = int(np.argmax(np.abs(vec)))
    axis_sign = 1 if vec[axis_idx] >= 0 else -1
    axis_str = ['x', 'y', 'z'][axis_idx]
    if axis_sign < 0:
        axis_str = f'-{axis_str}'
    distance = np.abs(vec[axis_idx])
    # Make sure distance is not ridiculously tiny
    distance = max(distance, 0.10)
    return axis_str, distance


def run_combined_drawer_disposal():
    print("\n===== [Task] Combined Drawer‑Opening + Rubbish Disposal =====")
    env, task = setup_environment()
    try:
        # ---------- Reset environment ----------
        descriptions, obs = task.reset()

        # -------------- Recording ---------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(
            task.get_observation)

        # -------- Retrieve relevant positions --------
        positions = get_object_positions()

        drawer_side_pos   = np.array(positions['bottom_side_pos'])
        drawer_anchor_pos = np.array(positions['bottom_anchor_pos'])
        drawer_joint_pos  = np.array(positions['bottom_joint_pos'])

        rubbish_pos       = np.array(positions['rubbish'])
        bin_pos           = np.array(positions['bin'])

        # ==========================================================
        # STEP‑1  Rotate gripper (zero_deg ➔ ninety_deg)
        # ==========================================================
        target_quat = _quat_from_euler(0, 0, 90)   # yaw 90°
        obs, reward, done = rotate(env, task,
                                   target_quat=target_quat)
        if done:
            print("[Task] Finished early after rotate.")
            return

        # ==========================================================
        # STEP‑2  Move to drawer side‑pos
        # ==========================================================
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos)
        if done:
            print("[Task] Finished early after move‑to‑side.")
            return

        # ==========================================================
        # STEP‑3  Move to anchor‑pos (handle)
        # ==========================================================
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Finished early after move‑to‑anchor.")
            return

        # ==========================================================
        # STEP‑4  Grasp the drawer handle
        # ==========================================================
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_axis='z')          # approach from above
        if done:
            print("[Task] Finished early after pick‑drawer.")
            return

        # ==========================================================
        # STEP‑5  Pull to open the drawer fully
        # ==========================================================
        pull_axis, pull_dist = _axis_and_distance(drawer_anchor_pos,
                                                  drawer_joint_pos)
        obs, reward, done = pull(env, task,
                                 pull_distance=pull_dist,
                                 pull_axis=pull_axis)
        if done:
            print("[Task] Finished early after pull (drawer open).")
            return

        # ==========================================================
        # STEP‑6  Pick up the rubbish from the table
        # ==========================================================
        obs, reward, done = pick(env, task,
                                 target_pos=rubbish_pos,
                                 approach_axis='z')
        if done:
            print("[Task] Finished early after pick‑rubbish.")
            return

        # ==========================================================
        # STEP‑7  Place the rubbish into the bin
        # ==========================================================
        obs, reward, done = place(env, task,
                                  target_pos=bin_pos,
                                  approach_axis='z')
        # Task finished – success assumed irrespective of the
        # environment’s ‘done’ flag here.
        print("[Task] Completed specification plan. "
              f"Done flag: {done}, Final reward: {reward}")

    except Exception as e:
        print("[Task] Exception occurred:", e)
        raise

    finally:
        shutdown_environment(env)
        print("===== [Task] Environment shutdown complete =====")


if __name__ == "__main__":
    run_combined_drawer_disposal()