# run_combined_task.py
#
# Goal:
#   1) Rotate the gripper 90 deg so that it can grasp the drawer handle.
#   2) Move to the drawer’s side­‑position, then to the anchor‑position.
#   3) Pick (grasp) the drawer handle and pull it to open the drawer.
#   4) Pick the rubbish from the table.
#   5) Place the rubbish in the bin – tomatoes are not touched.
#
# NOTE:
#   – Only predefined skills from `skill_code` are used.
#   – All imports requested by the skeleton are kept as‑is.

import numpy as np

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _safe_pos(positions, *keys):
    """Return the first key that exists in the positions dictionary."""
    for k in keys:
        if k in positions:
            return np.asarray(positions[k])
    raise KeyError(f"None of the keys {keys} found in object‑positions!")


def run_combined_task():
    print("=====  Combined Drawer‑and‑Trash Task  =====")

    # ------------------------------------------------------------------
    #  Environment set‑up
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset and obtain an initial observation
        descriptions, obs = task.reset()

        # ------------------------------------------------------------------
        #  (Optional) enable video recording
        # ------------------------------------------------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Get all relevant object positions
        # ------------------------------------------------------------------
        positions = get_object_positions()

        bottom_side_pos   = _safe_pos(positions, 'bottom_side_pos')
        bottom_anchor_pos = _safe_pos(positions, 'bottom_anchor_pos')
        rubbish_pos       = _safe_pos(positions, 'rubbish')
        bin_pos           = _safe_pos(positions, 'bin')

        # ------------------------------------------------------------------
        #  Step‑by‑step plan (matches the given specification order)
        # ------------------------------------------------------------------

        # --- Step‑1 : ROTATE (gripper → 90° around the Z‑axis) -------------
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi / 4.0), np.cos(np.pi / 4.0)])
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Early‑Exit] Task ended during rotate."); return

        # --- Step‑2 : MOVE to the drawer’s side‑position -------------------
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Early‑Exit] Task ended during move‑to‑side."); return

        # --- Step‑3 : MOVE to the drawer’s anchor‑position -----------------
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Early‑Exit] Task ended during move‑to‑anchor."); return

        # --- Step‑4 : PICK the drawer handle (grasp) -----------------------
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos,
                                 approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Early‑Exit] Task ended during pick‑drawer."); return

        # --- Step‑5 : PULL the drawer open --------------------------------
        #   Pull 20 cm straight along the +X axis (heuristic choice)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Early‑Exit] Task ended during pull‑drawer."); return

        # --- Step‑6 : PICK the rubbish off the table ----------------------
        obs, reward, done = pick(env, task, target_pos=rubbish_pos,
                                 approach_distance=0.15, approach_axis='z')
        if done:
            print("[Early‑Exit] Task ended during pick‑rubbish."); return

        # --- Step‑7 : PLACE the rubbish into the bin ----------------------
        obs, reward, done = place(env, task, target_pos=bin_pos,
                                  approach_distance=0.15, approach_axis='z')
        if done:
            print("[SUCCESS] Finished task after placing rubbish. Reward:", reward)
        else:
            print("[INFO] Plan finished but environment reports done=False.")

    except Exception as e:
        # Print any unexpected error for easier debugging
        print("Exception occurred during task execution:", repr(e))
        raise
    finally:
        # Always shut down the environment properly
        shutdown_environment(env)

    print("=====  End of Combined Task  =====")


if __name__ == "__main__":
    run_combined_task()
