# run_task_combined.py
#
# This script instantiates the RLBench environment, then executes the
# following high‑level plan (taken from the Specification):
#
#   1) rotate – rotate the gripper to 90 deg (so that it can approach the
#      drawer handle from the side)
#   2) move   – go to the “side” waypoint of the bottom drawer
#   3) move   – go to the “anchor” waypoint of the bottom drawer
#   4) pick   – grasp the drawer handle
#   5) pull   – pull the drawer open
#   6) move   – move above the “rubbish” object lying on the table
#   7) pick   – grasp the rubbish
#   8) move   – move above the bin
#   9) place  – drop the rubbish into the bin
#
# The script relies ONLY on the predefined skills contained in
# `skill_code.py`; no new primitives are introduced.

import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === import predefined skills (do NOT redefine them) ===
from skill_code import rotate, move, pick, pull, place


def compute_relative_quaternion(current_quat_xyzw, axis='z', angle_deg=90.0):
    """
    Returns a quaternion that is the result of rotating the current quaternion
    by `angle_deg` degrees about the specified axis (‘x’, ‘y’, or ‘z’).
    """
    base = R.from_quat(current_quat_xyzw)
    if axis == 'x':
        delta = R.from_euler('x', angle_deg, degrees=True)
    elif axis == 'y':
        delta = R.from_euler('y', angle_deg, degrees=True)
    else:
        delta = R.from_euler('z', angle_deg, degrees=True)
    return (delta * base).as_quat()   # SciPy returns [x, y, z, w]


def safe_get(positions, key, default=None):
    """
    Utility wrapper around dict access that raises a descriptive error if the
    key does not exist (helps with debugging mis‑spelled object names).
    """
    if key not in positions:
        raise KeyError(f"[Task] Key ‘{key}’ not found in object_positions "
                       f"(available keys: {list(positions.keys())}).")
    return positions[key]


def run_combined_task():
    print("===== Starting Combined Drawer‑and‑Disposal Task =====")

    env, task = setup_environment()
    try:
        # Reset the environment and grab the initial observation
        descriptions, obs = task.reset()

        # ------------------------------------------------------------------
        # (Optional) video recording helpers
        # ------------------------------------------------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # Retrieve all relevant 3‑D positions from the helper module
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer‑related positions
        drawer_side_pos   = safe_get(positions, 'bottom_side_pos')
        drawer_anchor_pos = safe_get(positions, 'bottom_anchor_pos')

        # Rubbish and bin
        # The observation file lists “item1/2/3”; we assume ‘item3’ is the
        # rubbish that must go to the bin.  Adjust here if your environment
        # uses a different naming scheme.
        rubbish_pos = safe_get(positions, 'item3')
        bin_pos     = safe_get(positions, 'bin')

        # ------------------------------------------------------------------
        # Step‑by‑step execution of the oracle plan
        # ------------------------------------------------------------------

        # 1) rotate – turn gripper so that it is aligned side‑ways
        print("\n[Plan‑Step 1] rotate gripper 90° around z‑axis")
        current_quat = obs.gripper_pose[3:7]  # [x, y, z, w]
        target_quat  = compute_relative_quaternion(current_quat,
                                                   axis='z',
                                                   angle_deg=90.0)
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Episode finished during ‘rotate’.")
            return

        # 2) move – to drawer side waypoint
        print("\n[Plan‑Step 2] move to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, np.array(drawer_side_pos))
        if done:
            print("[Task] Episode finished during move‑to‑side.")
            return

        # 3) move – to drawer anchor waypoint
        print("\n[Plan‑Step 3] move to drawer anchor position:",
              drawer_anchor_pos)
        obs, reward, done = move(env, task, np.array(drawer_anchor_pos))
        if done:
            print("[Task] Episode finished during move‑to‑anchor.")
            return

        # 4) pick – grasp the drawer handle
        print("\n[Plan‑Step 4] pick the drawer handle at anchor")
        obs, reward, done = pick(env, task,
                                 target_pos=np.array(drawer_anchor_pos),
                                 approach_distance=0.12,
                                 approach_axis='z')
        if done:
            print("[Task] Episode finished during drawer pick.")
            return

        # 5) pull – pull the drawer outwards by 12 cm along +x
        print("\n[Plan‑Step 5] pull the drawer open by 0.12 m")
        obs, reward, done = pull(env, task,
                                 pull_distance=0.12,
                                 pull_axis='x')
        if done:
            print("[Task] Episode finished during pull.")
            return

        # 6) move – above the rubbish lying on the table
        print("\n[Plan‑Step 6] move above rubbish at:", rubbish_pos)
        obs, reward, done = move(env, task, np.array(rubbish_pos))
        if done:
            print("[Task] Episode finished during move‑to‑rubbish.")
            return

        # 7) pick – grasp the rubbish
        print("\n[Plan‑Step 7] pick the rubbish")
        obs, reward, done = pick(env, task,
                                 target_pos=np.array(rubbish_pos),
                                 approach_distance=0.12,
                                 approach_axis='z')
        if done:
            print("[Task] Episode finished during rubbish pick.")
            return

        # 8) move – above the bin
        print("\n[Plan‑Step 8] move above the bin at:", bin_pos)
        obs, reward, done = move(env, task, np.array(bin_pos))
        if done:
            print("[Task] Episode finished during move‑to‑bin.")
            return

        # 9) place – drop the rubbish inside the bin
        print("\n[Plan‑Step 9] place the rubbish into the bin")
        obs, reward, done = place(env, task,
                                  target_pos=np.array(bin_pos),
                                  approach_distance=0.12,
                                  approach_axis='z')
        if done:
            print("[Task] Episode finished during place (completed).")
        else:
            print("[Task] Plan executed, but ‘done’ is False – "
                  "task may require additional verification.")

    finally:
        shutdown_environment(env)
        print("===== Environment shut down – task script finished =====")


if __name__ == "__main__":
    run_combined_task()
