# run_oracle_task.py
#
# This script executes the oracle plan that satisfies the task:
# “Select a drawer and open it fully, then pick up the rubbish and leave it in the trash can.”
#
# The plan follows the Specification steps exactly, using only predefined
# skills from skill_code (rotate, move, pull, pick, place).

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape               # noqa: F401  (kept for completeness / env internals)
from pyrep.objects.proximity_sensor import ProximitySensor   # noqa: F401

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_oracle_task():
    print("==========  START ORACLE TASK  ==========")

    # ------------------------------------------------------------------
    # 1) Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video recording utilities
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2) Fetch all known object positions from the helper module
        # ------------------------------------------------------------------
        positions = get_object_positions()

        def pos(name: str) -> np.ndarray:
            """Helper: fetch object position or raise a descriptive error."""
            if name not in positions:
                raise KeyError(f"[Oracle] ‼ Position for '{name}' not found. "
                               f"Available keys: {list(positions.keys())}")
            return np.asarray(positions[name], dtype=np.float32)

        # Decide which object will act as “rubbish”
        rubbish_name = "rubbish" if "rubbish" in positions else "item1"

        # ------------------------------------------------------------------
        # 3) Oracle Plan — executes the Specification steps
        # ------------------------------------------------------------------

        # STEP‑1  rotate  → turn the gripper to 90 deg about Z (ninety_deg)
        quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, quat_90deg_z)
        if done:
            print("[Oracle] Finished early at rotate.")  # unlikely
            return

        # STEP‑2  move    → bottom_side_pos (side position of bottom drawer)
        obs, reward, done = move(env, task, pos("bottom_side_pos"))
        if done:
            return

        # STEP‑3  move    → bottom_anchor_pos (anchor/handle of bottom drawer)
        obs, reward, done = move(env, task, pos("bottom_anchor_pos"))
        if done:
            return

        # STEP‑4  pull    → open the drawer along +X by ~0.20 m
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            return

        # STEP‑5  move    → retreat to a safe waypoint (if available)
        safe_wp = "waypoint1" if "waypoint1" in positions else "bottom_side_pos"
        obs, reward, done = move(env, task, pos(safe_wp))
        if done:
            return

        # STEP‑6  pick    → grab the rubbish on the table
        obs, reward, done = pick(env, task, pos(rubbish_name))
        if done:
            return

        # STEP‑7  place   → drop rubbish into the bin
        obs, reward, done = place(env, task, pos("bin"))
        if done:
            print("[Oracle] 🎉  Task completed successfully!  Reward:", reward)
        else:
            print("[Oracle] Task finished (done flag False).")

    except Exception as err:
        print("[Oracle] ❌  Exception during execution:", err)

    finally:
        # Always cleanly shut down to free simulation resources
        shutdown_environment(env)

    print("==========  END ORACLE TASK  ==========")


if __name__ == "__main__":
    run_oracle_task()
