import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape

# === RLBench helper imports (provided externally, DO NOT MODIFY) ===
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Skill functions (already implemented, DO NOT REDEFINE) ===
from skill_code import move, pick, place, rotate, pull

# -------------------------------------------------------------
# Utility helpers
# -------------------------------------------------------------
def quat_from_euler_xyz(rx=0.0, ry=0.0, rz=0.0, degrees=True):
    """Return xyzw quaternion for given xyz Euler angles."""
    return R.from_euler('xyz', [rx, ry, rz], degrees=degrees).as_quat()

def fetch_position(name, cache):
    """
    Safely fetch an object's world position. Falls back to Shape if
    the name is missing in the supplied cache.
    """
    if name in cache:
        return np.asarray(cache[name])
    try:
        return np.asarray(Shape(name).get_position())
    except Exception as e:
        raise RuntimeError(f"[Task] Cannot resolve position for '{name}': {e}")

# -------------------------------------------------------------
# Main task runner
# -------------------------------------------------------------
def run_open_drawer_and_dispose_rubbish():
    """
    Executes the oracle plan described in the natural‑language goal /
    specification:
        1) Rotate gripper to 90°.
        2) Move to bottom drawer side position.
        3) Move to bottom drawer anchor (handle).
        4) Grasp the drawer handle.
        5) Pull drawer open.
        6) Pick every rubbish object on the table.
        7) Place rubbish into the bin.
    """
    print("========== [Task] START ==========")

    # ------------------------------------------------------------------
    #  Environment setup (RLBench)
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the task and grab the initial observation
        descriptions, obs = task.reset()

        # (Optional) video recording
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------
        # Gather all useful 3‑D positions
        # ------------------------------------------------------------------
        pos_cache = get_object_positions()  # external util returns dict

        bottom_side_pos   = fetch_position('bottom_side_pos', pos_cache)
        bottom_anchor_pos = fetch_position('bottom_anchor_pos', pos_cache)
        bin_pos           = fetch_position('bin', pos_cache)

        # Rubbish / item list (could be empty or missing – handle gracefully)
        candidate_items = ['item1', 'item2', 'item3', 'rubbish']
        rubbish_items = []
        for name in candidate_items:
            try:
                rubbish_items.append((name, fetch_position(name, pos_cache)))
            except RuntimeError:
                # Ignore missing objects
                continue
        if not rubbish_items:
            print("[Task] WARNING: No rubbish objects found! Task will only open drawer.")

        # ------------------------------------------------------------------
        #  STEP‑BY‑STEP EXECUTION  (follow provided specification plan)
        # ------------------------------------------------------------------

        # Step‑1 : rotate  gripper  → 90° about world‑Z
        ninety_quat = quat_from_euler_xyz(rz=90.0)   # xyz euler (0,0,90°)
        print("\n[Plan] Step‑1  Rotate gripper to 90°.")
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_quat,
            max_steps=120,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during rotation.")
            return

        # Step‑2 : move to side position of the bottom drawer
        print("\n[Plan] Step‑2  Move to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(
            env, task,
            target_pos=bottom_side_pos,
            max_steps=120,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move to side position.")
            return

        # Step‑3 : move to anchor (handle) position
        print("\n[Plan] Step‑3  Move to bottom_anchor_pos (handle):", bottom_anchor_pos)
        obs, reward, done = move(
            env, task,
            target_pos=bottom_anchor_pos,
            max_steps=120,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move to anchor position.")
            return

        # Step‑4 : pick (grasp) drawer handle
        print("\n[Plan] Step‑4  Grasp drawer handle.")
        obs, reward, done = pick(
            env, task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.06,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during handle grasp.")
            return

        # Step‑5 : pull drawer open (along world‑X, 20 cm outward)
        print("\n[Plan] Step‑5  Pull drawer open (0.20 m along +x).")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.20,
            pull_axis='x',     # positive‑x; adjust if required by env
            max_steps=120,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during drawer pull.")
            return

        # Optional: release the handle so that gripper becomes free
        print("\n[Plan] Releasing drawer handle.")
        obs, reward, done = place(
            env, task,
            target_pos=bottom_anchor_pos,     # same spot – just open gripper
            approach_distance=0.04,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during handle release.")
            return

        # ------------------------------------------------------------------
        #  Dispose of rubbish (iterate through all found items)
        # ------------------------------------------------------------------
        for obj_name, obj_pos in rubbish_items:
            print(f"\n[Plan] Pick {obj_name} at table: {obj_pos}")
            # Move above object (safety waypoint slightly above)
            waypoint_above_obj = obj_pos + np.array([0, 0, 0.10])
            obs, reward, done = move(
                env, task,
                target_pos=waypoint_above_obj,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print(f"[Task] Terminated while moving above {obj_name}.")
                return

            obs, reward, done = pick(
                env, task,
                target_pos=obj_pos,
                approach_distance=0.06,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Task] Terminated during pick of {obj_name}.")
                return

            # Move above bin then place
            waypoint_above_bin = bin_pos + np.array([0, 0, 0.10])
            print(f"[Plan] Move to bin to dispose {obj_name}.")
            obs, reward, done = move(
                env, task,
                target_pos=waypoint_above_bin,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print(f"[Task] Terminated while moving above bin with {obj_name}.")
                return

            print(f"[Plan] Place {obj_name} into bin.")
            obs, reward, done = place(
                env, task,
                target_pos=bin_pos,
                approach_distance=0.06,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Task] Terminated during place of {obj_name}.")
                return

        # ------------------------------------------------------------------
        #  Task completed!
        # ------------------------------------------------------------------
        print("\n========== [Task] COMPLETED SUCCESSFULLY ==========")

    finally:
        shutdown_environment(env)
        print("========== [Environment] Shutdown complete ==========")


# -------------------------------------------------------------
# Entry point
# -------------------------------------------------------------
if __name__ == "__main__":
    run_open_drawer_and_dispose_rubbish()
