import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# ===== Import all predefined skills =====
from skill_code import rotate, move, pick, pull, place


def run_skeleton_task():
    """
    Task description:
      1. Rotate the gripper so that it can approach a drawer handle from the side.
      2. Move to the side‑position of the (un‑locked) bottom drawer.
      3. Move to the drawer anchor (handle) position.
      4. Grasp the handle (pick).
      5. Pull the drawer open.
      6. Move to the rubbish object lying on the table.
      7. Pick the rubbish.
      8. Move to the bin.
      9. Place the rubbish in the bin.
    """
    print("\n==========  STARTING TASK  ==========")

    # ------------------------------------------------------------------
    # 1)  Environment & helpers
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # reset RLBench task
        descriptions, obs = task.reset()

        # optional video recording wrapper
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2)  Gather all important object poses
        # ------------------------------------------------------------------
        positions = get_object_positions()        # returns dict[str, np.ndarray]

        # Fail‑early sanity checks – tell user exactly what’s missing
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'item3',          # we treat “item3” as the rubbish
            'bin'
        ]
        for key in required_keys:
            if key not in positions:
                raise RuntimeError(
                    f"[run_skeleton_task] '{key}' missing from object_positions."
                    " Make sure the scene defines every required waypoint."
                )

        side_pos   = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['item3']
        bin_pos     = positions['bin']

        # ------------------------------------------------------------------
        # 3)  Execute the plan (follows the given 9‑step oracle plan)
        # ------------------------------------------------------------------

        # STEP‑1  ── rotate gripper 90° about z
        print("\n--- Step‑1 : rotate ---")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()   # xyzw
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished after rotate – exiting early.")
            return

        # STEP‑2  ── move to side position of bottom drawer
        print("\n--- Step‑2 : move (to side‑pos) ---")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑3  ── move to anchor (handle) position
        print("\n--- Step‑3 : move (to anchor‑pos) ---")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑4  ── pick the drawer handle
        print("\n--- Step‑4 : pick (handle) ---")
        obs, reward, done = pick(env, task, anchor_pos, approach_axis='-z')
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑5  ── pull the drawer open (along +x by 0.20 m)
        print("\n--- Step‑5 : pull drawer ---")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑6  ── move above rubbish lying on the table
        print("\n--- Step‑6 : move (to rubbish) ---")
        obs, reward, done = move(env, task, rubbish_pos + np.array([0, 0, 0.10]))
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑7  ── pick the rubbish
        print("\n--- Step‑7 : pick (rubbish) ---")
        obs, reward, done = pick(env, task, rubbish_pos, approach_axis='-z')
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑8  ── move above the bin
        print("\n--- Step‑8 : move (to bin) ---")
        obs, reward, done = move(env, task, bin_pos + np.array([0, 0, 0.15]))
        if done:
            print("[Task] Finished unexpectedly – exiting.")
            return

        # STEP‑9  ── place rubbish inside the bin
        print("\n--- Step‑9 : place (in bin) ---")
        obs, reward, done = place(env, task, bin_pos, approach_axis='-z')
        if done:
            print("[Task] Task signalled done=True after place.")

        print("\n==========  TASK COMPLETED  ==========")

    except Exception as e:
        # Any unhandled exception should still close the environment gracefully
        print(f"[run_skeleton_task] Exception raised: {e}")
        raise

    finally:
        shutdown_environment(env)


if __name__ == "__main__":
    run_skeleton_task()
