import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Execute the oracle plan:
        1) Rotate the gripper to the ‘ninety deg’ pose.
        2) Move to the bottom drawer’s side position.
        3) Pick (grasp) the drawer handle at its anchor position.
        4) Pull the drawer open.
        5) Pick the rubbish (item3) from the table.
        6) Place the rubbish into the bin.
    """
    print("===== Starting Skeleton Task =====")

    # -------------------------------------------------
    #  Environment setup
    # -------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # optional video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # -------------------------------------------------
        #  Query all relevant object positions
        # -------------------------------------------------
        positions = get_object_positions()
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'item3', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[Task] Required object key '{k}' missing in get_object_positions().")

        side_pos          = positions['bottom_side_pos']
        anchor_pos        = positions['bottom_anchor_pos']
        rubbish_pos       = positions['item3']     # treat item3 as the rubbish object
        bin_pos           = positions['bin']

        # -------------------------------------------------
        #  STEP 1 – Rotate gripper to ninety deg
        # -------------------------------------------------
        # Create a quaternion equivalent to (roll, pitch, yaw) = (0, -90°, 0)
        ninety_quat = R.from_euler('xyz', [0.0, -np.pi / 2, 0.0]).as_quat()
        print("[Plan] Step 1 – rotate gripper to 90 deg orientation")
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_quat,
            max_steps=120,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Episode finished during rotate.")
            return

        # -------------------------------------------------
        #  STEP 2 – Move to the drawer’s side position
        # -------------------------------------------------
        print("[Plan] Step 2 – move to bottom drawer side‑position")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            max_steps=120,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Episode finished during move to side‑pos.")
            return

        # -------------------------------------------------
        #  STEP 3 – Pick (grasp) the drawer handle
        # -------------------------------------------------
        print("[Plan] Step 3 – pick drawer handle at anchor‑position")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.12,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode finished while grasping drawer.")
            return

        # -------------------------------------------------
        #  STEP 4 – Pull the drawer open
        # -------------------------------------------------
        # Pull 0.15 m along +x (typical drawer direction)
        print("[Plan] Step 4 – pull drawer outwards")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=120,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Episode finished during drawer pull.")
            return

        # -------------------------------------------------
        #  STEP 5 – Pick the rubbish from the table
        # -------------------------------------------------
        print("[Plan] Step 5 – pick the rubbish (item3)")
        obs, reward, done = pick(
            env, task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode finished while picking rubbish.")
            return

        # -------------------------------------------------
        #  STEP 6 – Place the rubbish into the bin
        # -------------------------------------------------
        print("[Plan] Step 6 – place rubbish into bin")
        obs, reward, done = place(
            env, task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )

        # -------------------------------------------------
        #  Summary
        # -------------------------------------------------
        if done:
            print("[Task] Goal achieved! Reward:", reward)
        else:
            print("[Task] Finished all actions; environment did not signal done.")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
