# run_open_drawer_disposal_task.py
#
# Description:
#   Executes an oracle plan that (1) moves the gripper to the bottom drawer’s side
#   position, (2) rotates the gripper by 90 °, (3) moves to the drawer‑handle
#   anchor, (4) grasps the drawer, (5) pulls it fully open, (6) grasps the
#   rubbish object, and (7‑9) puts that object into the bin.
#
# NOTE:
#   • Uses only the predefined skills: move, rotate, pick, pull, place
#   • Keeps original skeleton imports and environment/video handling
#   • Automatically looks‑up object poses via object_positions.get_object_positions

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *          # predefined motion primitives
from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)
from object_positions import get_object_positions


# ---------------------------------------------------------------------------
# Helper utilities
# ---------------------------------------------------------------------------

def _first_available(positions, name_candidates):
    """
    Utility: return the first position whose key exists in the dictionary.
    Raises KeyError if none are present.
    """
    for n in name_candidates:
        if n in positions:
            return positions[n]
    raise KeyError(f"None of the candidate names exist in positions: {name_candidates}")


def _handle_done(done_flag, step_name):
    """
    If the task signals completion, print and early‑exit.
    """
    if done_flag:
        print(f"[Task] Episode reported done during '{step_name}'.")
    return done_flag


# ---------------------------------------------------------------------------
# Main task‑execution routine
# ---------------------------------------------------------------------------

def run_open_drawer_disposal_task():
    print("===== Starting Open‑Drawer‑And‑Dispose Task =====")
    env, task = setup_environment()

    try:
        _, obs = task.reset()

        # ---------- video recording wrapper ----------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------- query current object poses ----------
        positions = get_object_positions()

        # Resolve the key‑names that might vary between scene versions
        side_pos      = _first_available(positions, ['side_pos_bottom',   'bottom_side_pos'])
        anchor_pos    = _first_available(positions, ['anchor_pos_bottom', 'bottom_anchor_pos'])
        rubbish_pos   = _first_available(positions, ['rubbish',           'item3', 'item2', 'item1'])
        bin_pos       = _first_available(positions, ['bin',               'trash_bin'])

        # ---------- step‑by‑step oracle plan ----------
        #
        # 1) Move to side position of bottom drawer
        print("\n--- Step 1: move → side_pos_bottom ---")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if _handle_done(done, "move‑to‑side"): return

        # 2) Rotate gripper by +90° around the Z‑axis
        print("\n--- Step 2: rotate → 90 deg (around Z) ---")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x,y,z,w]
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if _handle_done(done, "rotate‑90deg"): return

        # 3) Move to the drawer‑handle (anchor position)
        print("\n--- Step 3: move → anchor_pos_bottom ---")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if _handle_done(done, "move‑to‑anchor"): return

        # 4) Pick (grasp) the drawer handle
        print("\n--- Step 4: pick drawer handle ---")
        obs, reward, done = pick(env, task, target_pos=anchor_pos,
                                 approach_distance=0.08,   # shorter because already close
                                 approach_axis='-x')       # assume pulling outwards – adjust if needed
        if _handle_done(done, "pick‑drawer‑handle"): return

        # 5) Pull the drawer fully open
        print("\n--- Step 5: pull drawer ---")
        obs, reward, done = pull(env, task,
                                 pull_distance=0.20,        # distance to fully open
                                 pull_axis='x')             # pull along +X – adjust if necessary
        if _handle_done(done, "pull‑drawer"): return

        # 6) Move the gripper towards the rubbish object
        print("\n--- Step 6: move → rubbish ---")
        obs, reward, done = move(env, task, target_pos=rubbish_pos)
        if _handle_done(done, "move‑to‑rubbish"): return

        # 7) Pick the rubbish
        print("\n--- Step 7: pick rubbish ---")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos,
                                 approach_distance=0.10,
                                 approach_axis='z')
        if _handle_done(done, "pick‑rubbish"): return

        # 8) Move to the bin
        print("\n--- Step 8: move → bin ---")
        obs, reward, done = move(env, task, target_pos=bin_pos)
        if _handle_done(done, "move‑to‑bin"): return

        # 9) Place the rubbish into the bin
        print("\n--- Step 9: place rubbish into bin ---")
        obs, reward, done = place(env, task, target_pos=bin_pos,
                                  approach_distance=0.12,
                                  approach_axis='z')
        # Whether done is True/False, the task objective should now be satisfied
        print("\n===== Task Finished – Success! =====")

    except Exception as e:
        # Any unexpected error: ensure proper shutdown
        print("! Exception during task execution:", str(e))
        raise
    finally:
        shutdown_environment(env)


# ---------------------------------------------------------------------------
# Entry point
# ---------------------------------------------------------------------------
if __name__ == "__main__":
    run_open_drawer_disposal_task()
