# run_open_drawer_disposal_task.py

import numpy as np
from scipy.spatial.transform import Rotation as R

from pyrep.objects.shape import Shape              # (kept – required by skeleton)
from pyrep.objects.proximity_sensor import ProximitySensor   # (kept – required by skeleton)

from env import setup_environment, shutdown_environment

# NOTE: all low‑level skills already exist in skill_code, we simply import them.
from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_open_drawer_disposal_task():
    """
    Executes the oracle plan specified in the task description:

        1.  Rotate the gripper 90° (zero_deg → ninety_deg)
        2.  Move to bottom drawer side position
        3.  Move to bottom drawer anchor position
        4.  Grasp the drawer handle (pick)
        5.  Pull the drawer open
        6.  Pick the rubbish from the table
        7.  Place the rubbish inside the bin
    """
    print("===== [Task] Open Drawer & Dispose Rubbish =====")

    # ------------------------------------------------------------------
    #  Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset to the initial state of the scenario
        descriptions, obs = task.reset()

        # ---- Optional video recording helpers ----
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        # ------------------------------------------

        # ------------------------------------------------------------------
        #  Retrieve object poses that are required for the oracle plan
        # ------------------------------------------------------------------
        positions = get_object_positions()      # dict[str, np.ndarray(3,)]

        #
        # Make sure the positions we need actually exist
        #
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'rubbish', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[Task] Required key '{k}' missing from object_positions!")

        bottom_side_pos   = np.asarray(positions['bottom_side_pos'], dtype=float)
        bottom_anchor_pos = np.asarray(positions['bottom_anchor_pos'], dtype=float)
        rubbish_pos       = np.asarray(positions['rubbish'], dtype=float)
        bin_pos           = np.asarray(positions['bin'], dtype=float)

        # ------------------------------------------------------------------
        #  Oracle Plan Execution
        # ------------------------------------------------------------------

        # ------------------------------------------------------------
        # Step‑1  rotate(gripper, zero_deg, ninety_deg)
        # ------------------------------------------------------------
        print("\n--- Step 1: Rotate gripper 90° ---")
        obs = task.get_observation()
        current_quat = np.asarray(obs.gripper_pose[3:7], dtype=float)
        #
        # Rotate about +Z axis by 90°.  We construct the target quaternion
        # in the world frame by composing the current orientation with Rz(90°)
        #
        additional_rot = R.from_euler('z', 90, degrees=True).as_quat()
        target_quat = R.from_quat(current_quat) * R.from_quat(additional_rot)
        target_quat = target_quat.as_quat()      # xyzw
        obs, reward, done = rotate(
            env, task, target_quat,
            max_steps=150, threshold=0.03, timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 1!")
            return

        # ------------------------------------------------------------
        # Step‑2  move‑to‑side(gripper, bottom, nowhere‑pos, side‑pos‑bottom)
        # ------------------------------------------------------------
        print("\n--- Step 2: Move to bottom‑drawer side position ---")
        obs, reward, done = move(
            env, task, bottom_side_pos,
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 2!")
            return

        # ------------------------------------------------------------
        # Step‑3  move‑to‑anchor(gripper, bottom, side‑pos‑bottom, anchor‑pos‑bottom)
        # ------------------------------------------------------------
        print("\n--- Step 3: Move to bottom‑drawer anchor position ---")
        obs, reward, done = move(
            env, task, bottom_anchor_pos,
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 3!")
            return

        # ------------------------------------------------------------
        # Step‑4  pick‑drawer(≈pick) at anchor‑pos‑bottom
        # ------------------------------------------------------------
        print("\n--- Step 4: Grasp drawer handle ---")
        #
        # Use the generic 'pick' skill on the drawer‑handle location.
        # A small approach distance keeps the trajectory local.
        #
        obs, reward, done = pick(
            env, task, bottom_anchor_pos,
            approach_distance=0.05, max_steps=150,
            threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 4!")
            return

        # ------------------------------------------------------------
        # Step‑5  pull(gripper, bottom)
        # ------------------------------------------------------------
        print("\n--- Step 5: Pull drawer open ---")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.20,          # 20 cm pull
            pull_axis='x',               # typically drawers open along +X
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 5!")
            return

        # ------------------------------------------------------------
        # Step‑6  pick(rubbish, table)
        # ------------------------------------------------------------
        print("\n--- Step 6: Pick up rubbish ---")
        obs, reward, done = pick(
            env, task, rubbish_pos,
            approach_distance=0.15, max_steps=150,
            threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Finished unexpectedly after Step 6!")
            return

        # ------------------------------------------------------------
        # Step‑7  place(rubbish, bin)
        # ------------------------------------------------------------
        print("\n--- Step 7: Place rubbish into bin ---")
        obs, reward, done = place(
            env, task, bin_pos,
            approach_distance=0.15, max_steps=150,
            threshold=0.01, approach_axis='z', timeout=10.0
        )

        # ------------------------------------------------------------------
        #  Final result
        # ------------------------------------------------------------------
        if done:
            print("[Task] SUCCESS: Goal achieved!  Reward:", reward)
        else:
            print("[Task] Plan finished; task not flagged as done (done=False).")
    finally:
        shutdown_environment(env)

    print("===== [Task] Finished =====")


if __name__ == "__main__":
    run_open_drawer_disposal_task()
