import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Import predefined skill functions ===
# NOTE: the following import brings in pick, place, move, rotate, pull, etc.
from skill_code import pick, place, move, rotate, pull


def run_skeleton_task():
    """Execute the oracle plan for:
         1) Rotating the gripper,
         2) Opening the bottom drawer (if not locked),
         3) Throwing the rubbish into the bin.
    """
    print("\n===== Starting Combined Task =====")

    # ------------------------------------------------------------------
    #  Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()

    try:
        # Reset and fetch the very first observation
        descriptions, obs = task.reset()

        # Optionally start video recording
        init_video_writers(obs)

        # Wrap step / get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Retrieve static object positions
        # ------------------------------------------------------------------
        # Expected keys (see provided object list):
        #   bottom_side_pos, bottom_anchor_pos, rubbish, bin, ...
        positions = get_object_positions()

        # Grace‑check that we have the required objects
        required_keys = [
            "bottom_side_pos",
            "bottom_anchor_pos",
            "rubbish",
            "bin",
        ]
        for key in required_keys:
            if key not in positions:
                raise KeyError(
                    f"[Task] Required object '{key}' not found in object_positions dict."
                )

        bottom_side_pos = np.asarray(positions["bottom_side_pos"])
        bottom_anchor_pos = np.asarray(positions["bottom_anchor_pos"])
        rubbish_pos = np.asarray(positions["rubbish"])
        bin_pos = np.asarray(positions["bin"])

        # ------------------------------------------------------------------
        #  STEP‑BY‑STEP PLAN (Specification compliant)
        # ------------------------------------------------------------------
        #
        #  Spec steps:
        #   1) rotate(gripper, zero_deg → ninety_deg)
        #   2) move‑to‑side(gripper, bottom, nowhere‑pos → side‑pos‑bottom)
        #   3) move‑to‑anchor(gripper, bottom, side‑pos‑bottom → anchor‑pos‑bottom)
        #   4) pick‑drawer(gripper, bottom, anchor‑pos‑bottom)
        #   5) pull(gripper, bottom)
        #   6) pick(rubbish, table)
        #   7) place(rubbish, bin)
        #
        #  We map those symbolic actions to available skills:
        #   • rotate  -> rotate(...)
        #   • move‑to‑*  -> move(...)
        #   • pick‑drawer -> pick(...)  (target = drawer anchor handle)
        #   • pull -> pull(...)
        #   • pick / place -> pick(...), place(...)
        #
        #  Note: we use reasonable numeric parameters when the spec gives none.
        # ------------------------------------------------------------------

        # --------------------------------------------------------------
        # 1) Rotate gripper from zero_deg to ninety_deg about Z axis
        # --------------------------------------------------------------
        print("\n--- [1] Rotate gripper to 90° about Z ---")
        target_quat = R.from_euler("z", 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished prematurely after rotate.")
            return

        # --------------------------------------------------------------
        # 2) Move to the drawer side position
        # --------------------------------------------------------------
        print("\n--- [2] Move gripper to bottom_side_pos ---")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Finished prematurely after move‑to‑side.")
            return

        # --------------------------------------------------------------
        # 3) Move from side‑pos to anchor‑pos (handle)
        # --------------------------------------------------------------
        print("\n--- [3] Move gripper to bottom_anchor_pos (handle) ---")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Finished prematurely after move‑to‑anchor.")
            return

        # --------------------------------------------------------------
        # 4) Grasp the drawer handle (pick‑drawer)
        #     • We treat the drawer handle as an object located at
        #       bottom_anchor_pos. Use a very small approach distance so
        #       we do not overshoot inside the drawer.
        # --------------------------------------------------------------
        print("\n--- [4] Grasp drawer handle ---")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.05,   # short approach, we are already close
            approach_axis="z",
        )
        if done:
            print("[Task] Finished prematurely after pick‑drawer.")
            return

        # --------------------------------------------------------------
        # 5) Pull the drawer straight out along the X axis
        #     • We assume the drawer opens along positive X.
        #     • A modest pull distance (e.g., 0.10 m) is chosen.
        # --------------------------------------------------------------
        print("\n--- [5] Pull the drawer open ---")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.10,
            pull_axis="x",
        )
        if done:
            print("[Task] Finished prematurely after pull.")
            return

        # --------------------------------------------------------------
        # 6) Pick up the rubbish from the table
        # --------------------------------------------------------------
        print("\n--- [6] Pick the rubbish from the table ---")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[Task] Finished prematurely after picking rubbish.")
            return

        # --------------------------------------------------------------
        # 7) Place the rubbish into the bin
        # --------------------------------------------------------------
        print("\n--- [7] Place the rubbish into the bin ---")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            approach_axis="z",
        )

        # ------------------------------------------------------------------
        #  Final status
        # ------------------------------------------------------------------
        if done:
            print("\n*****  Task completed successfully! *****\n")
        else:
            print(
                "\n*****  Task finished sequence but environment did not report "
                "done=True.  *****\n"
            )

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
        raise

    finally:
        # Always ensure the environment shuts down
        shutdown_environment(env)

    print("===== End of Combined Task =====")


if __name__ == "__main__":
    run_skeleton_task()
