import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Pre‑implemented skills ===
from skill_code import rotate, move, pick, pull, place


def run_robot_disposal_with_drawer_task():
    """
    1) Rotate the gripper so it can grasp the drawer handle from the side.
    2) Move to the side waypoint of the bottom drawer.
    3) Move to the anchor (handle) position of the bottom drawer.
    4) Close the gripper on the handle (pick).
    5) Pull the drawer open.
    6) Pick the rubbish that is on the table.
    7) Place the rubbish in the bin.
    """
    print("\n=================  START TASK  =================")

    # ------------------------------------------------------------------
    #  Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the RLBench task
        descriptions, obs = task.reset()

        # Optional video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)            # wrap step for recording
        task.get_observation = recording_get_observation(  # wrap get_observation
            task.get_observation
        )

        # ------------------------------------------------------------------
        #  Fetch all relevant object positions from RLBench
        # ------------------------------------------------------------------
        print("[Info] Retrieving object positions …")
        pos_dict = get_object_positions()

        # Drawer way‑points
        side_pos = pos_dict.get("bottom_side_pos")
        anchor_pos = pos_dict.get("bottom_anchor_pos")

        # Rubbish to be picked and the target bin
        rubbish_pos = pos_dict.get("rubbish")
        bin_pos = pos_dict.get("bin")

        # Sanity‑check that everything exists
        for name, p in [
            ("bottom_side_pos", side_pos),
            ("bottom_anchor_pos", anchor_pos),
            ("rubbish", rubbish_pos),
            ("bin", bin_pos),
        ]:
            if p is None:
                raise RuntimeError(
                    f"[Error] Required key “{name}” missing from object_positions."
                )

        # Convert any list/tuple into np.array for the skill functions
        side_pos = np.asarray(side_pos, dtype=np.float32)
        anchor_pos = np.asarray(anchor_pos, dtype=np.float32)
        rubbish_pos = np.asarray(rubbish_pos, dtype=np.float32)
        bin_pos = np.asarray(bin_pos, dtype=np.float32)

        # ------------------------------------------------------------------
        #  Step‑by‑step execution (matches the 7‑step specification)
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # 1) rotate
        #    – rotate 90 ° about the world Z‑axis so that the gripper finger
        #      faces the drawer handle correctly.
        print("\n[Step 1] rotate gripper 90 ° about Z")
        target_quat = R.from_euler("xyz", [0, 0, np.pi / 2]).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished early after rotate.")
            return

        # 2) move – go to side waypoint of bottom drawer
        print("\n[Step 2] move to bottom_side_pos:", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Finished early after move‑to‑side.")
            return

        # 3) move – slide in towards the anchor (handle)
        print("\n[Step 3] move to bottom_anchor_pos:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Finished early after move‑to‑anchor.")
            return

        # 4) pick – close on the drawer handle
        print("\n[Step 4] pick drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.05,  # short distance – we are already close
            approach_axis="-z",      # descend slightly onto the handle
        )
        if done:
            print("[Task] Finished early during pick‑drawer.")
            return

        # 5) pull – pull the drawer straight out along +X by ~0.20 m
        print("\n[Step 5] pull drawer open")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,
            pull_axis="x",
        )
        if done:
            print("[Task] Finished early during drawer pull.")
            return

        # Optional: release the drawer handle by opening the gripper a little
        #           (place the ‘object’ exactly where it is – just opens fingers)
        # We can simply call place at the same anchor position with zero approach
        print("[Info] Releasing drawer handle")
        _ = place(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            approach_axis="-z",
        )

        # 6) pick – collect the rubbish from the table
        print("\n[Step 6] move & pick rubbish on table")
        # Move above rubbish first for safety
        obs, reward, done = move(env, task, rubbish_pos + np.array([0, 0, 0.15]))
        if done:
            print("[Task] Finished early while approaching rubbish.")
            return
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.12,
            approach_axis="-z",
        )
        if done:
            print("[Task] Finished early during pick‑rubbish.")
            return

        # 7) place – drop the rubbish in the bin
        print("\n[Step 7] place rubbish in the bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.12,
            approach_axis="-z",
        )

        # ------------------------------------------------------------------
        #  Task completed!
        # ------------------------------------------------------------------
        if done:
            print(f"[Success] Task finished!  Reward: {reward}")
        else:
            print("[Info] The task episode has not signalled completion (done=False).")

    finally:
        # Always shut down the environment, no matter what happens
        shutdown_environment(env)
        print("\n=================  END TASK  =================")


if __name__ == "__main__":
    run_robot_disposal_with_drawer_task()