import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# ----- Pre–defined skills (imported exactly as supplied) -----
from skill_code import rotate, move, pick, pull, place


def _vector_to_axis(vec: np.ndarray) -> str:
    """Convert a 3‑D direction vector into one of the axis strings
    recognised by the `pull` primitive (‘x’, ‘-x’, ‘y’, …)."""
    if np.linalg.norm(vec) < 1e-6:
        # No meaningful direction – default to +x
        return 'x'
    idx = np.argmax(np.abs(vec))
    sign = '' if vec[idx] >= 0 else '-'
    return f'{sign}{"xyz"[idx]}'


def run_skeleton_task():
    print("===== Starting Drawer‑and‑Dispose Task =====")

    # ============ Environment initialisation ============
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # optional: start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ============ Retrieve all object positions ============
        positions = get_object_positions()
        # Defensive check
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos',
            'rubbish', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"Object position for '{k}' not found!")

        # ============ Oracle plan execution ============
        done = False
        reward = 0.0

        # ----- Step‑1 : rotate gripper from zero_deg to ninety_deg -----
        # The target quaternion is a +90° rotation around the Z‑axis.
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # (x,y,z,w)
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Early‑Exit] Task finished during rotate.")
            return

        # ----- Step‑2 : move‑to‑side (bottom drawer side handle position) -----
        side_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Early‑Exit] Task finished during move‑to‑side.")
            return

        # ----- Step‑3 : move‑to‑anchor (align with drawer handle) -----
        anchor_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Early‑Exit] Task finished during move‑to‑anchor.")
            return

        # ----- Step‑4 : pick‑drawer (grasp the drawer handle) -----
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.10,
            approach_axis='y'    # typical drawer handles are grasped from +y/‑y
        )
        if done:
            print("[Early‑Exit] Task finished during pick‑drawer.")
            return

        # ----- Step‑5 : pull the drawer open -----
        joint_pos = positions['bottom_joint_pos']
        pull_vec = joint_pos - anchor_pos
        pull_dist = np.linalg.norm(pull_vec)
        pull_axis = _vector_to_axis(pull_vec)
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_dist,
            pull_axis=pull_axis
        )
        if done:
            print("[Early‑Exit] Task finished during pull.")
            return

        # Drawer is now open, hand becomes free automatically in many RLBench
        # tasks after pull; if not, that will still be okay for the subsequent
        # pick because pick() closes the gripper again.

        # ----- Step‑6 : pick the rubbish from the table -----
        rubbish_pos = positions['rubbish']
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done:
            print("[Early‑Exit] Task finished during pick‑rubbish.")
            return

        # ----- Step‑7 : place rubbish in the bin -----
        bin_pos = positions['bin']
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            approach_axis='z'
        )

        # ============ Task concluded ============
        if done:
            print("[Success] Goal achieved! Final reward:", reward)
        else:
            print("[Info] Plan executed but environment did not report done=True.")

    finally:
        shutdown_environment(env)
        print("===== Environment shut down – End of Task =====")


if __name__ == "__main__":
    run_skeleton_task()
