# run_combined_task.py
#
# Description:
#  - Executes the oracle plan described in the specification:
#       1) Rotate gripper 90 deg
#       2) Move to the drawer’s side position
#       3) Move to the drawer’s anchor position
#       4) Grip the drawer handle
#       5) Pull the drawer open
#       6) Pick the rubbish on the table
#       7) Place the rubbish in the bin
#  - Uses ONLY the predefined skills contained in skill_code
#  - Records the episode (video helpers) and shuts the simulator
#    down cleanly in every circumstance

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
# Import every registered skill exactly as provided
from skill_code import rotate, move, pull, pick, place

from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)

from object_positions import get_object_positions


def run_combined_task():
    """Runs the full sequence for the ‘pull‑drawer & dispose rubbish’ task."""
    print("===== Starting Combined Drawer + Disposal Task =====")

    # -------------------------------------------------
    #  Environment initialisation
    # -------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the task (RLBench typical signature: returns descriptions, obs)
        descriptions, obs = task.reset()

        # Optional video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)                    # wrap step
        task.get_observation = recording_get_observation(        # wrap get_obs
            task.get_observation)

        # -------------------------------------------------
        #  Obtain positions of every relevant object
        # -------------------------------------------------
        # object_positions.py should return a dict mapping the names (strings)
        # in the provided “Object List” to 3‑D numpy vectors.
        positions = get_object_positions()

        # Helper: fetch a position, raise meaningful error if missing
        def pos(name: str) -> np.ndarray:
            if name not in positions:
                raise KeyError(f"[Task] Position for '{name}' not found "
                               f"in object_positions dictionary.")
            return np.asarray(positions[name], dtype=np.float32)

        # Frequently accessed positions
        bottom_side      = pos("bottom_side_pos")
        bottom_anchor    = pos("bottom_anchor_pos")
        rubbish_pos      = pos("rubbish")
        bin_pos          = pos("bin")

        # -------------------------------------------------
        #  ORACLE PLAN EXECUTION
        # -------------------------------------------------
        done = False
        reward = 0.0

        # STEP‑1 ─ ROTATE gripper from 0° → 90° about Z
        if not done:
            target_quat = R.from_euler('z', 90.0, degrees=True).as_quat()
            print("\n[Step‑1] rotate → ninety_deg (90° about Z)")
            obs, reward, done = rotate(
                env, task,
                target_quat=target_quat,
                max_steps=120,
                threshold=0.03,
                timeout=10.0
            )

        # STEP‑2 ─ MOVE to the bottom drawer’s side position
        if not done:
            print("\n[Step‑2] move → bottom_side_pos")
            obs, reward, done = move(
                env, task,
                target_pos=bottom_side,
                max_steps=120,
                threshold=0.01,
                timeout=10.0
            )

        # STEP‑3 ─ MOVE to the bottom drawer’s anchor position
        if not done:
            print("\n[Step‑3] move → bottom_anchor_pos")
            obs, reward, done = move(
                env, task,
                target_pos=bottom_anchor,
                max_steps=120,
                threshold=0.01,
                timeout=10.0
            )

        # STEP‑4 ─ PICK the drawer handle (acts as “pick‑drawer”)
        if not done:
            print("\n[Step‑4] pick‑drawer (grip handle at anchor pos)")
            obs, reward, done = pick(
                env, task,
                target_pos=bottom_anchor,
                approach_distance=0.10,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

        # STEP‑5 ─ PULL the drawer open
        if not done:
            print("\n[Step‑5] pull drawer outwards")
            # Empirically, positive X is usually the drawer’s pull direction
            PULL_DIST  = 0.20   # metres
            PULL_AXIS  = 'x'    # adjust if environment axes differ
            obs, reward, done = pull(
                env, task,
                pull_distance=PULL_DIST,
                pull_axis=PULL_AXIS,
                max_steps=120,
                threshold=0.01,
                timeout=10.0
            )

        # STEP‑5b ─ RELEASE the handle (open the gripper)
        #          We re‑use the “place” skill at the same position so the
        #          gripper opens (place() ends by opening the fingers).
        if not done:
            print("\n[Step‑5b] release drawer handle (open gripper)")
            obs, reward, done = place(
                env, task,
                target_pos=bottom_anchor,     # no real movement
                approach_distance=0.00,       # stay in place
                max_steps=30,
                threshold=0.005,
                approach_axis='z',
                timeout=5.0
            )

        # STEP‑6 ─ PICK the rubbish from the table
        if not done:
            print("\n[Step‑6] pick rubbish on table")
            obs, reward, done = pick(
                env, task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

        # STEP‑7 ─ PLACE the rubbish in the bin
        if not done:
            print("\n[Step‑7] place rubbish in bin")
            obs, reward, done = place(
                env, task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=120,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

        # -------------------------------------------------
        #  SUCCESS INFO
        # -------------------------------------------------
        if done:
            print(f"[Task] Episode finished early by environment (reward={reward}).")
        else:
            print("[Task] Oracle plan executed completely.")
            print(f"[Task] Final reward = {reward}")

    except Exception as e:
        # Robust error report plus guaranteed cleanup
        print("[Task] *** Exception occurred! ***")
        print(e)
        raise

    finally:
        shutdown_environment(env)
        print("===== Environment shut down. =====")


if __name__ == "__main__":
    run_combined_task()
