# run_skeleton_task.py (Filled-in Version)

import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# import every low-level skill already provided in “skill_code”
from skill_code import *          # noqa: F403,F401  (predefined skills only)

from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

# helper that simply returns a dict {object_name : np.ndarray(x,y,z)}
from object_positions import get_object_positions


# --------------------------------------------------------------------------- #
# Utility / helper functions
# --------------------------------------------------------------------------- #
def _open_gripper(env, task, n_steps: int = 15):
    """Force the gripper to open (to guarantee the predicate ‘handempty’)."""
    obs = task.get_observation()
    quat = obs.gripper_pose[3:7]           # keep the current orientation
    action = np.zeros(env.action_shape)
    action[3:7] = quat
    action[-1] = 1.0                       # +1 → open
    for _ in range(n_steps):
        obs, reward, done = task.step(action)
        if done:
            break
    return obs, reward, done


def _close_gripper(env, task, n_steps: int = 15):
    """Close the gripper (useful for some skills that expect grasp closed)."""
    obs = task.get_observation()
    quat = obs.gripper_pose[3:7]
    action = np.zeros(env.action_shape)
    action[3:7] = quat
    action[-1] = -1.0                      # −1 → close
    for _ in range(n_steps):
        obs, reward, done = task.step(action)
        if done:
            break
    return obs, reward, done


def ensure_hand_empty(env, task):
    """
    “Exploration” helper:
      – Tries to guarantee that the gripper really is empty / open.
      – Returns True when we believe the predicate (handempty) now holds.
    """
    try:
        obs = task.get_observation()
        # Many RLBench tasks expose ‘gripper_open’ (0: closed, 1: opened)
        # We fall back to simply trying to open if the key is missing.
        gripper_open = obs.misc.get("gripper_open", None)
        if gripper_open is not None and gripper_open > 0.9:
            return True
    except Exception:
        # If “misc” or the key doesn’t exist we just try opening.
        pass

    _open_gripper(env, task, n_steps=10)
    return True


# --------------------------------------------------------------------------- #
# Main entry point
# --------------------------------------------------------------------------- #
def run_skeleton_task():
    """
    Generic runnable script that:
      1. Boots the RLBench environment.
      2. Performs a brief “exploration” phase (ensuring ‘handempty’ holds).
      3. Executes a minimal pick-and-place routine using only pre-defined
         skills from `skill_code`.  This demonstrates that the missing
         predicate has been handled and that the provided skills can now
         operate without failure.
    """
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------ #
    # 1) Environment setup
    # ------------------------------------------------------------------ #
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)            # (recording helpers)

        # Wrap step / get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # -------------------------------------------------------------- #
        # 2) “Exploration” – make sure predicate ‘handempty’ holds
        # -------------------------------------------------------------- #
        print("[Exploration] Ensuring gripper is empty (handempty).")
        ensure_hand_empty(env, task)
        print("[Exploration] handempty predicate should now hold.\n")

        # -------------------------------------------------------------- #
        # 3) Retrieve object positions (we only need one object here).
        # -------------------------------------------------------------- #
        positions = get_object_positions()     # dict: name → np.array(3)
        if not positions:
            print("[Warning] No objects were detected by get_object_positions()!")
            print("          Nothing to do.  Exiting early.")
            return

        # Pick the first object in the dictionary (arbitrary choice)
        obj_name, obj_pos = next(iter(positions.items()))
        print(f"[Task] Target object chosen: {obj_name} @ {obj_pos}")

        # Compute a naïve place location: +20 cm on the x-axis
        place_pos = obj_pos + np.array([0.20, 0.0, 0.0])
        print(f"[Task] Chosen place location: {place_pos}")

        # -------------------------------------------------------------- #
        # 4) Execute high-level plan using predefined skills
        # -------------------------------------------------------------- #

        # ---- PICK ---------------------------------------------------- #
        print(f"\n[Action] pick( {obj_name} )")
        try:
            obs, reward, done = pick(     # noqa: F405  (imported via *)
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.15,
                max_steps=120,
                threshold=0.015,
                approach_axis="z",
                timeout=10.0,
            )
        except Exception as e:
            # If pick failed due to hand not empty, open and retry once
            print(f"[pick] Exception caught: {e}")
            print("[pick] Re-ensuring handempty and retrying once.")
            ensure_hand_empty(env, task)
            obs, reward, done = pick(     # noqa: F405
                env,
                task,
                target_pos=obj_pos,
                approach_distance=0.15,
                max_steps=120,
                threshold=0.015,
                approach_axis="z",
                timeout=10.0,
            )

        if done:
            print("[Task] Episode finished unexpectedly during pick!")
            return

        # ---- PLACE --------------------------------------------------- #
        print(f"\n[Action] place( {obj_name} )")
        obs, reward, done = place(        # noqa: F405
            env,
            task,
            target_pos=place_pos,
            approach_distance=0.15,
            max_steps=120,
            threshold=0.015,
            approach_axis="z",
            timeout=10.0,
        )

        # We intentionally ignore the final ‘done’ flag; shutdown in finally
        if done:
            print("[Task] Episode finished right after place.")

        print("\n===== Task Finished Successfully =====")

    finally:
        # Always shutdown the environment even when an exception occurs
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()