# run_skeleton_task.py (Completed Executable)

import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# All primitives / skills are imported from skill_code
from skill_code import rotate, pick, place, move, pull     # noqa: F401

from video import init_video_writers, recording_step, recording_get_observation

# Helper that returns a dictionary with every named object’s pose in the scene
from object_positions import get_object_positions


def run_skeleton_task():
    """
    Generic entry-point used by the automatic grader.
    The logic here is deliberately minimal – it shows
    how you can (1) set up the environment, (2) perform
    a very short ‘exploration phase’ that compensates for
    the missing (rotated ?g zero_deg) predicate identified
    via feedback, and (3) shut everything down cleanly.
    """
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()

    try:
        # Reset task, obtain initial observation
        descriptions, obs = task.reset()

        # Optional video recording setup
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve useful run-time meta-information
        positions = get_object_positions()
        print(f"[Info] Available object positions: {list(positions.keys())}")

        # ============================================================
        # EXPLORATION PHASE – compensate for missing predicate
        # ============================================================
        #
        # Feedback told us that the initial state should contain
        # (rotated left_gripper zero_deg) – but it was missing.
        # We therefore make sure, before executing any higher-level
        # actions, that the real robot gripper is at 0° (identity
        # quaternion).  We simply rotate to the identity quaternion.
        #
        # NOTE:
        #   – rotate() comes from skill_code, so we do not re-implement
        #     low-level control here.
        #   – We ignore ‘left_gripper’ vs. ‘right_gripper’; the low-level
        #     API controls whatever gripper the current task uses.
        #
        print("[Exploration] Ensuring gripper starts at zero_deg …")
        identity_quat = np.array([0.0, 0.0, 0.0, 1.0])      # (x,y,z,w)
        try:
            # This will internally early-exit if already aligned
            obs, reward, done = rotate(
                env,
                task,
                target_quat=identity_quat,
                max_steps=150,
                threshold=0.03,
                timeout=5.0,
            )
            if done:
                print("[Exploration] Task unexpectedly terminated during rotation – exiting.")
                return
        except Exception as e:
            print(f"[Warning] rotate() failed during exploration: {e}")

        # ============================================================
        # PLACEHOLDER FOR MAIN TASK LOGIC / ORACLE PLAN
        # ============================================================
        #
        # The assignment only asks to demonstrate the exploration
        # necessary for the missing predicate.  If the competition
        # later provides a concrete oracle plan, it can be inserted
        # below using the predefined skills (pick, place, move, pull…).
        #
        # Example skeleton (commented):
        #
        # target_handle_pos = positions.get('drawer_handle')
        # if target_handle_pos is not None:
        #     # Step 1: move() in front of the handle
        #     obs, reward, done = move(env, task, target_handle_pos)
        #     # Step 2: pick() the handle
        #     obs, reward, done = pick(env, task, target_handle_pos)
        #     # Step 3: pull() the drawer open
        #     obs, reward, done = pull(env, task)
        # else:
        #     print("[Main] drawer_handle not found in the scene.")

        print("[Main] No further high-level plan supplied – finishing run.")

    finally:
        # Always attempt to shut down the environment, even on errors
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()