# run_skeleton_task.py  (Completed version)

import inspect
import traceback

import numpy as np
from pyrep.objects.shape import Shape                      # kept – may be useful by the env
from pyrep.objects.proximity_sensor import ProximitySensor  # kept – may be useful by the env

from env import setup_environment, shutdown_environment
from skill_code import *                                   # do **not** redefine skills – use them directly
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ------------------------------------------------------------------------------------
# Helper: only pass the arguments that the particular skill function really accepts
# ------------------------------------------------------------------------------------
def _safe_call(func, **all_kwargs):
    try:
        sig = inspect.signature(func)
        filtered = {k: v for k, v in all_kwargs.items() if k in sig.parameters}
        # Make sure **required** positional/keyword parameters are present.
        # If something mandatory is missing we just let Python raise – we will catch it.
        return func(**filtered)
    except Exception as exc:
        print(f"[Warning] Calling skill {func.__name__} failed:\n    {exc}")
        traceback.print_exc()
        return None, 0.0, False     # compatible default (obs, reward, done)


# ------------------------------------------------------------------------------------
# Helper: simple static-analysis to discover which predicate is missing between
#         the exploration domain description and the combined main domain.
# ------------------------------------------------------------------------------------
def _detect_missing_predicate():
    # predicates explicitly present in the *combined* domain
    domain_predicates = {
        'at', 'holding', 'handempty',
        'is-locked', 'is-open',
        'rotated', 'gripper-at', 'holding-drawer',
        'is-side-pos', 'is-anchor-pos'
    }

    # predicates that appear in the *exploration* domain description
    exploration_predicates = {
        'robot-at', 'identified', 'temperature-known',
        'weight-known', 'durability-known', 'lock-known'
    }

    missing = exploration_predicates.difference(domain_predicates)
    print("[Analysis] Predicates that exist in the exploration domain but not in the "
          "combined domain:", missing)
    if 'lock-known' in missing:
        print("[Result] The critical missing predicate is: lock-known")


# ------------------------------------------------------------------------------------
#   Main entry point that follows the original skeleton logic
# ------------------------------------------------------------------------------------
def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------
    #  Environment initialisation
    # ------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()          # RLBench style reset()
        init_video_writers(obs)                  # video helpers

        # wrap step / get_obs with recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------------------------------------------
        #  Exploration phase – identify missing predicate
        # --------------------------------------------------
        _detect_missing_predicate()

        # --------------------------------------------------
        #  Retrieve all object positions from the helper
        # --------------------------------------------------
        positions = get_object_positions() or {}
        print(f"[Info] Environment currently reports {len(positions)} objects.")

        # identify drawers that actually exist in the scene
        drawer_priority = ['bottom', 'middle', 'top']
        drawer_candidates = [name for name in positions if 'drawer' in name.lower()]

        # re-order the drawers according to our desired priority (bottom → top)
        ordered_drawers = []
        for key in drawer_priority:
            for d in drawer_candidates:
                if key in d and d not in ordered_drawers:
                    ordered_drawers.append(d)
        # append any other drawers that did not match explicit keywords
        ordered_drawers.extend([d for d in drawer_candidates if d not in ordered_drawers])

        print(f"[Info] Detected drawers in scene   : {drawer_candidates}")
        print(f"[Info] Attempt order for drawers   : {ordered_drawers}")

        # --------------------------------------------------
        #  Main task logic – open the first available drawer
        # --------------------------------------------------
        for drawer_name in ordered_drawers:
            if drawer_name not in positions:
                print(f"[Skip] Drawer <{drawer_name}> is not in current object list – skipping.")
                continue

            drawer_pos = positions[drawer_name]
            print(f"\n[Task] Attempting to open drawer <{drawer_name}> located at {drawer_pos}")

            # 1) move end-effector close to the drawer handle
            _safe_call(
                move,
                env=env,
                task=task,
                target_pos=drawer_pos,
                approach_distance=0.12,
                max_steps=150,
                threshold=0.02,
                approach_axis='z',
                timeout=10.0
            )

            # 2) rotate the gripper if rotation skill exists / expects the argument list
            _safe_call(
                rotate,
                env=env,
                task=task,
                direction='clockwise',
                angle=np.deg2rad(90)        # 90° → radians if required
            )

            # 3) pull the drawer
            _safe_call(
                pull,
                env=env,
                task=task,
                distance=0.15,              # how far to pull – placeholder
                speed=0.05
            )

            # 4) after a successful pull we break – one open drawer is enough
            print(f"[Task] Finished interaction with <{drawer_name}> (attempted pull).")
            break
        else:
            print("[Warning] No available drawers were found – nothing to interact with.")

        print("===== End of Skeleton Task (normal shutdown) =====")

    finally:
        shutdown_environment(env)


# ----------------------------------------------------------
# Allow running directly
# ----------------------------------------------------------
if __name__ == "__main__":
    run_skeleton_task()