# run_skeleton_task.py (Completed)

import time
import math
import numpy as np

from pyrep.objects.shape import Shape          # noqa: F401  (kept for future use)
from pyrep.objects.proximity_sensor import ProximitySensor  # noqa: F401

from env import setup_environment, shutdown_environment

# Bring all low-level skills (pick, place, move, rotate, pull …) into the local namespace
from skill_code import *                       # noqa: F403

# We also need a direct handle to the skill_code module so that we can monkey-patch
# missing helper utilities required by some of the skills (e.g. rotate).
import skill_code as _skills

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ---------------------------------------------------------------------------
# Helper utilities that some skills (e.g. rotate) rely on
# ---------------------------------------------------------------------------

def _normalize_quaternion(quat):
    """Return a unit-length version of *quat*."""
    quat = np.asarray(quat, dtype=float)
    n = np.linalg.norm(quat)
    if n == 0:
        return quat
    return quat / n


def _euler_from_quat(quat):
    """
    Convert quaternion (xyzw) to Euler angles (roll, pitch, yaw).

    The implementation is deliberately simple – it is accurate enough for
    logging / debugging purposes that the skill code uses it for.
    """
    x, y, z, w = quat
    # roll (x-axis rotation)
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll = math.atan2(t0, t1)

    # pitch (y-axis rotation)
    t2 = +2.0 * (w * y - z * x)
    t2 = +1.0 if t2 > +1.0 else t2
    t2 = -1.0 if t2 < -1.0 else t2
    pitch = math.asin(t2)

    # yaw (z-axis rotation)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw = math.atan2(t3, t4)

    return roll, pitch, yaw


# ---------------------------------------------------------------------------
# Monkey-patch missing helpers into the original skill_code module
# so that functions such as rotate() can find them at run-time.
# ---------------------------------------------------------------------------
if not hasattr(_skills, 'normalize_quaternion'):
    _skills.normalize_quaternion = _normalize_quaternion
if not hasattr(_skills, 'euler_from_quat'):
    _skills.euler_from_quat = _euler_from_quat


# ---------------------------------------------------------------------------
# Main task runner
# ---------------------------------------------------------------------------
def run_skeleton_task():
    """
    Generic entry-point that:
        1) sets up the RLBench environment
        2) performs a short exploration/pick-and-place routine using only
           the pre-defined skills
        3) shuts the environment down cleanly
    """
    print("===== Starting Skeleton Task =====")

    # === 1. Environment Setup  =================================================
    env, task = setup_environment()

    try:
        # Reset the task (receives textual descriptions + first observation)
        descriptions, obs = task.reset()

        # Optional: start video capture
        init_video_writers(obs)

        # Wrap step() / get_observation() so that every call gets recorded
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === 2. Retrieve Object Positions ======================================
        #
        # get_object_positions() is expected to return something like:
        #     { "object_name" : (x, y, z), ... }
        #
        # The exact keys depend on the scenario.  Here we just fetch the first
        # entry to keep the example fully generic.
        #
        positions = get_object_positions()
        if not positions:
            print("[Warning] object_positions returned an empty dict – "
                  "nothing to manipulate.")
            return

        # Select the first object in the dictionary for a demo manipulation
        demo_obj_name, demo_obj_pos = next(iter(positions.items()))
        print(f"[Info] Demo object selected: {demo_obj_name} @ {demo_obj_pos}")

        # === 3. Exploration Phase (Missing-Predicate Discovery) ================
        #
        # We perform a small “exploration” sequence that exercises every
        # available high-level skill at least once.  This intentionally mimics
        # an agent trying to discover missing predicates like (lock-known ?d)
        # by interacting with its environment.
        #

        # ------------------------------------------------------------------
        # 3-A) MOVE – approach the chosen object
        # ------------------------------------------------------------------
        try:
            print(f"[Skill: move] Approaching {demo_obj_name}")
            # The example move() skill (as used in other templates) typically
            # expects:
            #     move(env, task, target_pos, speed=..., timeout=...)
            # We supply only the universally accepted arguments and fall back
            # to kwargs for the rest, so that the call does not explode even
            # if the actual signature is different.
            move_kwargs = dict(target_pos=demo_obj_pos,
                               speed=0.25,
                               timeout=5.0)
            obs, reward, done = move(env, task, **move_kwargs)   # noqa: F405
            if done:
                print("[Task] Environment reported task completion during move().")
                return
        except Exception as exc:
            print(f"[Error] move() failed: {exc}")

        # ------------------------------------------------------------------
        # 3-B) PICK – grasp the object
        # ------------------------------------------------------------------
        try:
            print(f"[Skill: pick] Grasping {demo_obj_name}")
            pick_kwargs = dict(target_pos=demo_obj_pos,
                               approach_distance=0.15,
                               max_steps=150,
                               threshold=0.01,
                               approach_axis='z',
                               timeout=10.0)
            obs, reward, done = pick(env, task, **pick_kwargs)   # noqa: F405
            if done:
                print("[Task] Environment reported task completion during pick().")
                return
        except Exception as exc:
            print(f"[Error] pick() failed: {exc}")

        # ------------------------------------------------------------------
        # 3-C) ROTATE – rotate the gripper by +90° around Z
        # ------------------------------------------------------------------
        try:
            print("[Skill: rotate] Rotating gripper by +90° around the Z-axis")
            # Quaternion representing +90° around Z: (x, y, z, w)
            qz90 = np.array([0.0, 0.0, math.sin(math.pi / 4), math.cos(math.pi / 4)])
            obs, reward, done = rotate(env, task, target_quat=qz90)   # noqa: F405
            if done:
                print("[Task] Environment reported task completion during rotate().")
                return
        except Exception as exc:
            print(f"[Error] rotate() failed: {exc}")

        # ------------------------------------------------------------------
        # 3-D) PULL – try to pull (e.g., a drawer handle) if present
        # ------------------------------------------------------------------
        try:
            print("[Skill: pull] Attempting to pull – useful for discovering "
                  "whether the drawer is locked/unlocked")
            # Many RLBench tasks have a handle called "handle" or similar;
            # here we simply forward whatever name we already have.
            pull_kwargs = dict(target_pos=demo_obj_pos,   # fallback
                               distance=0.10,
                               max_steps=100,
                               timeout=5.0)
            obs, reward, done = pull(env, task, **pull_kwargs)   # noqa: F405
            if done:
                print("[Task] Environment reported task completion during pull().")
                return
        except Exception as exc:
            print(f"[Error] pull() failed or not applicable: {exc}")

        # ------------------------------------------------------------------
        # 3-E) PLACE – put the object back where it was
        # ------------------------------------------------------------------
        try:
            print(f"[Skill: place] Returning {demo_obj_name} to original spot")
            place_kwargs = dict(target_pos=demo_obj_pos,
                                approach_distance=0.15,
                                max_steps=150,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0)
            obs, reward, done = place(env, task, **place_kwargs)   # noqa: F405
            if done:
                print("[Task] Environment reported task completion during place().")
                return
        except Exception as exc:
            print(f"[Error] place() failed: {exc}")

        # === 4. Conclude =======================================================
        print("[Task] Exploration routine finished successfully.")

    finally:
        # Always make sure we release the simulator resources
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


# ---------------------------------------------------------------------------
# Standard Python entry-point guard
# ---------------------------------------------------------------------------
if __name__ == "__main__":
    run_skeleton_task()