# run_skeleton_task.py (Completed)

import traceback
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# All primitive skills are expected to live in this module
from skill_code import pick, place, move, rotate, pull     # noqa: F401

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    """Generic skeleton for running any task in your simulation.

    The control logic below is deliberately *very* defensive.  During the
    exploration phase we attempt to execute the primitive skills in the most
    straight-forward way possible.  If an exception is raised that contains the
    word “handempty”, we immediately conclude that the root–cause of failure is
    the missing predicate `handempty` (see feedback) and we adapt our execution
    plan accordingly: we first free the gripper (place whatever it is currently
    holding) and then retry the intended pick action.

    NOTE:
        • No **new** skills are created – only the ones imported from
          `skill_code` are ever invoked.
        • All original imports of the skeleton remain untouched.
        • The code is written so that it can be executed in *any* RLBench-style
          task: we automatically discover the objects that exist in the current
          scene via `get_object_positions()` and then process them sequentially.
    """
    print("===== Starting Skeleton Task =====")

    # ----------------------------------------------------------------------
    # 1)  Environment setup
    # ----------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # ----- Video recording helpers (optional) --------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2)  Retrieve initial world-state information
        # ------------------------------------------------------------------
        positions = get_object_positions()          # {name: (x, y, z), ...}
        if not positions:
            print("[WARNING] No objects discovered by get_object_positions(). "
                  "Nothing to do.")
            return

        # We do not know which object is the “bin”.  A very common naming
        # convention in the RLBench tasks is to include the substring “bin”
        # or “goal”.  We search for it.  Fallback: pick the last entry.
        bin_name = None
        for key in positions:
            if "bin" in key.lower() or "goal" in key.lower():
                bin_name = key
                break
        if bin_name is None:
            # Arbitrary fallback – anything that is *not* the first element
            # to avoid bin==obj for single-object scenes.
            bin_name = list(positions.keys())[-1]

        bin_pos = positions[bin_name]

        # All manipulatable objects == anything that is *not* the bin
        manipulatable_objects = [k for k in positions if k != bin_name]

        # We maintain a *very* small amount of internal state that mirrors the
        # `handempty` predicate.  We start with the assumption that the gripper
        # is free right after task.reset().
        hand_empty = True
        holding_object_name = None

        # ------------------------------------------------------------------
        # 3)  Sequentially manipulate every object
        # ------------------------------------------------------------------
        for obj_name in manipulatable_objects:
            obj_pos = positions[obj_name]
            print(f"[Task] Processing object: {obj_name} @ {obj_pos}")

            # --------------------------------------------------------------
            # 3.1)  Exploration – try to PICK the object
            # --------------------------------------------------------------
            while True:
                try:
                    # 3.1.1)  Move close to the object
                    #         We pick an arbitrary approach distance + axis.
                    move(
                        env=env,
                        task=task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=150,
                        threshold=0.015,
                        approach_axis="z",
                        timeout=5.0,
                    )

                    # 3.1.2)  Try to pick it up
                    obs, reward, done = pick(
                        env=env,
                        task=task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=150,
                        threshold=0.01,
                        approach_axis="z",
                        timeout=5.0,
                    )

                    hand_empty = False
                    holding_object_name = obj_name
                    print(f"[Task] Successfully picked: {obj_name}")
                    break     # leave retry-loop

                except Exception as e:
                    # ------------------------------------------------------
                    # We treat any error mentioning “handempty” as the signal
                    # that the missing predicate caused the failure.  This is
                    # exactly the feedback we received from the oracle.
                    # ------------------------------------------------------
                    if "handempty" in str(e).lower():
                        print("[Exploration] Detected missing predicate "
                              "`handempty` – freeing gripper and retrying.")
                        if not hand_empty and holding_object_name is not None:
                            # We have something in the hand – place it first.
                            try:
                                place(
                                    env=env,
                                    task=task,
                                    target_pos=bin_pos,
                                    approach_distance=0.10,
                                    max_steps=150,
                                    threshold=0.02,
                                    approach_axis="z",
                                    timeout=5.0,
                                )
                                print(f"[Exploration] Placed temporary object "
                                      f"into bin: {holding_object_name}")
                            except Exception:
                                traceback.print_exc()
                            finally:
                                hand_empty = True
                                holding_object_name = None
                        else:
                            # According to our local state we should already be
                            # empty-handed.  Log & abort the retry to avoid an
                            # infinite loop.
                            print("[ERROR] hand_empty was already True but "
                                  "still got a handempty-related failure. "
                                  "Aborting.")
                            raise e
                        # …then retry the pick() call in the while-loop
                    else:
                        print("[ERROR] Unhandled exception in pick():")
                        traceback.print_exc()
                        raise e

            # --------------------------------------------------------------
            # 3.2)  Place the object into the bin / goal container
            # --------------------------------------------------------------
            try:
                move(
                    env=env,
                    task=task,
                    target_pos=bin_pos,
                    approach_distance=0.15,
                    max_steps=150,
                    threshold=0.015,
                    approach_axis="z",
                    timeout=5.0,
                )

                obs, reward, done = place(
                    env=env,
                    task=task,
                    target_pos=bin_pos,
                    approach_distance=0.10,
                    max_steps=150,
                    threshold=0.02,
                    approach_axis="z",
                    timeout=5.0,
                )
                hand_empty = True
                holding_object_name = None
                print(f"[Task] Successfully placed: {obj_name} into {bin_name}")

                # If the current RLBench task signals completion – stop early
                if done:
                    print("[Task] Task reported completion – exiting early.")
                    return

            except Exception as e:
                print("[ERROR] Failed during place():")
                traceback.print_exc()
                # Decide whether to continue or to abort the entire script
                # (here we abort – cannot safely recover)
                raise e

        print("[Task] All objects processed – task execution finished.")

    finally:
        # ------------------------------------------------------------------
        # 4) Always shut down the simulation – even when errors happen
        # ------------------------------------------------------------------
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()