# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull            # <- only use the predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _safe_call(skill_fn, *args, **kwargs):
    """
    Wrapper for safely calling a low-level skill.  Any exception that bubbles up
    from the underlying controller is caught so that the rest of the scenario
    can continue (or terminate gracefully) instead of crashing the entire run.
    """
    try:
        return skill_fn(*args, **kwargs)
    except Exception as exc:
        print(f"[Warning] Skill <{skill_fn.__name__}> failed with: {exc}")
        # The majority of rl-bench like primitives return a triple, so keep shape
        return None, 0.0, False


def _check_required_objects(required, actual_pos_dict):
    """
    Make sure every object we intend to touch exists in the environment.
    """
    missing = [obj for obj in required if obj not in actual_pos_dict]
    if missing:
        print(f"[Error] The following expected objects are missing in the "
              f"environment description: {missing}")
    return len(missing) == 0


def _calibrate_force_if_needed(task):
    """
    A placeholder for any additional safety / calibration.  Not every task
    requires explicit force limits, but we expose the hook in case the underlying
    implementation does.  (Nothing is done here because low-level access is not
    a part of this exercise.)
    """
    try:
        # Some tasks expose a 'calibrate_gripper_force' helper.
        if hasattr(task, "calibrate_gripper_force"):
            print("[Task] Performing force calibration for gripper …")
            task.calibrate_gripper_force()
    except Exception as exc:
        print(f"[Warning] Force calibration failed / skipped ({exc}).")


def run_skeleton_task():
    """Entry-point that sets up the world and executes a simple exploration +
    manipulation routine that is valid for the combined-domain example."""
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        # ---------------------------------------------------------------------
        # 0) Reset task & bring up video writers
        # ---------------------------------------------------------------------
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # ---------------------------------------------------------------------
        # 1) Wrap step() / get_observation() so every frame is recorded
        # ---------------------------------------------------------------------
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------------------------------
        # 2) Obtain a dictionary of all important object locations
        # ---------------------------------------------------------------------
        positions = get_object_positions()        # dict: name -> (x, y, z)
        print(f"[Debug] Known positions: {list(positions.keys())}")

        # ---------------------------------------------------------------------
        # 3) Ensure we have every object that we want to manipulate
        # ---------------------------------------------------------------------
        TABLE_LOC   = "table"
        BIN_LOC     = "bin"
        GRIPPER_OBJ = "gripper"

        # These are the objects required for the disposal part of the demo
        objects_of_interest = ["tomato1", "tomato2", "rubbish"]
        if not _check_required_objects(objects_of_interest + [TABLE_LOC, BIN_LOC,
                                                              GRIPPER_OBJ],
                                       positions):
            print("[Fatal] Missing required objects.  Aborting task.")
            return

        # ---------------------------------------------------------------------
        # 4) (Optional) Calibrate force / tactile sensors before heavy lifting
        # ---------------------------------------------------------------------
        _calibrate_force_if_needed(task)

        # ---------------------------------------------------------------------
        # 5) Simple EXPLORATION phase – identify and learn object properties
        #    The idea is to call 'move' so that, via the exploration domain, we
        #    trigger conditional effects such as (identified ?obj), etc.
        # ---------------------------------------------------------------------
        for target in [TABLE_LOC, BIN_LOC]:
            print(f"[Exploration] Moving robot base to '{target}' to identify "
                  "objects present at that location.")
            _safe_call(move,
                       env, task,
                       target_pos=positions[target],
                       approach_distance=0.20,
                       max_steps=100,
                       threshold=0.02,
                       approach_axis='z',
                       timeout=5.0)

        # ---------------------------------------------------------------------
        # 6) PICK-AND-PLACE for each object  (main manipulation logic)
        # ---------------------------------------------------------------------
        for obj_name in objects_of_interest:
            if obj_name not in positions:
                print(f"[Warning] '{obj_name}' does not exist, skipping.")
                continue

            obj_pos = positions[obj_name]
            print(f"[Task] Picking '{obj_name}' from the table …")

            # a) Move close to the object
            _safe_call(move,
                       env, task,
                       target_pos=obj_pos,
                       approach_distance=0.15,
                       max_steps=100,
                       threshold=0.01,
                       approach_axis='z',
                       timeout=5.0)

            # b) Pick it up  (this automatically sets weight-known / durability-known
            #    according to the exploration-domain specification)
            obs, reward, picked = _safe_call(pick,
                                             env, task,
                                             target_pos=obj_pos,
                                             approach_distance=0.05,
                                             max_steps=50,
                                             threshold=0.01,
                                             approach_axis='z',
                                             timeout=5.0)

            if not picked:
                print(f"[Warning] Pick failed for '{obj_name}', skipping place.")
                continue

            # c) Move to the bin
            _safe_call(move,
                       env, task,
                       target_pos=positions[BIN_LOC],
                       approach_distance=0.20,
                       max_steps=100,
                       threshold=0.02,
                       approach_axis='z',
                       timeout=5.0)

            # d) Place the object into the bin
            print(f"[Task] Placing '{obj_name}' into the bin …")
            _safe_call(place,
                       env, task,
                       target_pos=positions[BIN_LOC],
                       approach_distance=0.05,
                       max_steps=50,
                       threshold=0.01,
                       approach_axis='z',
                       timeout=5.0)

        # ---------------------------------------------------------------------
        # 7) DRAWER subsystem demonstration – open the bottom drawer
        # ---------------------------------------------------------------------
        print("[Drawer] Attempting to open the bottom drawer …")

        # (1) Rotate gripper to ninety_deg so that the `move-to-side` pre-condition
        #     is met.  If angles are mere strings, we still pass them through.
        _safe_call(rotate,
                   env, task,
                   gripper=GRIPPER_OBJ,
                   from_angle='zero_deg',
                   to_angle='ninety_deg')

        # (2) Move to the drawer’s side position
        _safe_call(move,
                   env, task,
                   target_pos=positions["side-pos-bottom"],
                   approach_distance=0.10,
                   max_steps=80,
                   threshold=0.01,
                   approach_axis='xy',
                   timeout=4.0)

        # (3) Move to drawer’s anchor position
        _safe_call(move,
                   env, task,
                   target_pos=positions["anchor-pos-bottom"],
                   approach_distance=0.05,
                   max_steps=80,
                   threshold=0.01,
                   approach_axis='xy',
                   timeout=4.0)

        # (4) Because there is no dedicated 'pick-drawer' primitive in the exposed
        #     Python API list (only 'pick' exists), we simply call pick on the
        #     drawer handle to satisfy (holding-drawer gripper bottom).
        _safe_call(pick,
                   env, task,
                   target_pos=positions["anchor-pos-bottom"],
                   approach_distance=0.02,
                   max_steps=30,
                   threshold=0.005,
                   approach_axis='xy',
                   timeout=3.0)

        # (5) Finally pull – which should transition (is-open bottom)
        _safe_call(pull,
                   env, task,
                   gripper=GRIPPER_OBJ,
                   drawer='bottom')

        print("[Drawer] Bottom drawer should now be open!")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()