# run_skeleton_task.py (Completed Implementation)

import re

import time

import math

import traceback

from typing import Set, List

import numpy as np

from pyrep.objects.shape import Shape          # kept – may be used by some skills

from pyrep.objects.proximity_sensor import ProximitySensor  # kept – may be used by some skills

from env import setup_environment, shutdown_environment

from skill_code import *                        # DO NOT redefine skills – we just import everything

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# -----------------------------------------------------------

# Utility helpers that were missing but are required by skills

# -----------------------------------------------------------

def normalize_quaternion(q: np.ndarray) -> np.ndarray:

    """Normalise a quaternion (x, y, z, w)."""

    q = np.asarray(q, dtype=np.float64)

    n = np.linalg.norm(q)

    if n == 0:

        return q

    return q / n

def euler_from_quat(q: np.ndarray):

    """

    Convert quaternion (x,y,z,w) to Euler angles (roll, pitch, yaw).

    The implementation follows standard aerospace sequences (XYZ / intrinsic).

    """

    x, y, z, w = q

    # roll (x-axis rotation)

    sinr_cosp = 2.0 * (w * x + y * z)

    cosr_cosp = 1.0 - 2.0 * (x * x + y * y)

    roll = math.atan2(sinr_cosp, cosr_cosp)

    # pitch (y-axis rotation)

    sinp = 2.0 * (w * y - z * x)

    if abs(sinp) >= 1:

        pitch = math.copysign(math.pi / 2.0, sinp)  # use 90° if out of range

    else:

        pitch = math.asin(sinp)

    # yaw (z-axis rotation)

    siny_cosp = 2.0 * (w * z + x * y)

    cosy_cosp = 1.0 - 2.0 * (y * y + z * z)

    yaw = math.atan2(siny_cosp, cosy_cosp)

    return roll, pitch, yaw

# -----------------------------------------------------------

# EXPLORATION PHASE – detect missing predicates

# -----------------------------------------------------------

EXPLORATION_DOMAIN_PDDL = """

(define (domain exploration)

  (:requirements :strips :typing :conditional-effects :universal-preconditions)

  (:types 

    robot object location

  )

  (:predicates

    (robot-at ?r - robot ?loc - location)

    (at ?obj - object ?loc - location)

    (identified ?obj - object)

    (temperature-known ?obj - object)

    (holding ?obj - object)

    (handempty)

    (weight-known ?obj - object)

    (durability-known ?obj - object)

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (identified ?obj)

        )

      )

    )

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (temperature-known ?obj)

        )

      )

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (weight-known ?obj)

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (durability-known ?obj)

    )

  )

  (:action pull

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (holding ?obj)

       (not (lock-known ?obj))

    )

    :effect (lock-known ?obj)

  )

)

"""

def _pddl_predicates_in_block(block: str) -> Set[str]:

    """Return predicate names declared in a predicate block."""

    # find every opening '(' followed by predicate name

    names = re.findall(r'\([\s]*([a-zA-Z0-9_-]+)', block)

    return set(names)

def detect_missing_predicates(pddl_text: str) -> Set[str]:

    """

    Very lightweight PDDL analyser:

    1) Collect predicate symbols declared inside the single (:predicates ...) section.

    2) Collect every symbol that appears as head of a list in the whole file.

    3) Return symbols that are used somewhere but never declared.

    """

    # 1. extract declared predicates

    declared: Set[str] = set()

    predicate_blocks: List[str] = re.findall(r'\(:predicates(.*?)\)', pddl_text, flags=re.S)

    if predicate_blocks:

        declared.update(_pddl_predicates_in_block(predicate_blocks[0]))

    # 2. extract every symbol that looks like a predicate anywhere

    used: Set[str] = set(re.findall(r'\([\s]*([a-zA-Z0-9_-]+)', pddl_text))

    # 3. Remove special keywords/operators that aren't predicates

    reserved = {

        'and', 'or', 'not', 'forall', 'when', 'imply',

        '=', '>', '<', '>=', '<=', '-',  # misc

        ':action', ':parameters', ':precondition', ':effect'

    }

    used -= reserved

    declared -= reserved

    # 4. missing

    missing = used - declared

    return missing

# -----------------------------------------------------------

# MAIN TASK PROCEDURE

# -----------------------------------------------------------

def run_skeleton_task():

    """Generic skeleton for running any task in the simulation."""

    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===

    env, task = setup_environment()

    try:

        # Reset the task to its initial state

        descriptions, obs = task.reset()

        # Optional: initialise video recording

        init_video_writers(obs)

        # Wrap step / get_observation for recording

        original_step = task.step

        task.step = recording_step(original_step)

        original_get_obs = task.get_observation

        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------

        # EXPLORATION PHASE – figure out which predicate is missing

        # ------------------------------------------------------------------

        print("\n[Exploration] Analysing exploration domain to detect missing predicates ...")

        missing_preds = detect_missing_predicates(EXPLORATION_DOMAIN_PDDL)

        if missing_preds:

            print(f"[Exploration] Detected predicate(s) referenced but not declared: {missing_preds}")

        else:

            print("[Exploration] No missing predicates detected!")

        # ------------------------------------------------------------------

        # BASIC DEMO PLAN – interact with the first object we find

        # (This is placeholder logic to demonstrate usage of imported skills)

        # ------------------------------------------------------------------

        positions = get_object_positions()  # returns {'object_name': (x,y,z), ...}

        if positions:

            # Pick the first object

            obj_name, obj_pos = next(iter(positions.items()))

            print(f"\n[Task] Demonstration: attempting to pick and place '{obj_name}' at position {obj_pos}")

            try:

                # 1) Pick

                obs, reward, done = pick(

                    env,

                    task,

                    target_pos=np.array(obj_pos),

                    approach_distance=0.15,

                    max_steps=120,

                    threshold=0.015,

                    approach_axis='z',

                    timeout=10.0

                )

                if done:

                    print("[Task] Episode finished during pick.")

                    return

                # Simple delay for clarity

                time.sleep(0.5)

                # 2) Place back to the same spot (for demo)

                obs, reward, done = place(

                    env,

                    task,

                    target_pos=np.array(obj_pos),

                    approach_distance=0.15,

                    max_steps=120,

                    threshold=0.015,

                    approach_axis='z',

                    timeout=10.0

                )

                if done:

                    print("[Task] Episode finished during place.")

                    return

                print("[Task] Demo pick-and-place complete.")

            except Exception as e:

                print("[Task] Exception encountered during demo actions!")

                traceback.print_exc()

        else:

            print("[Task] No object positions returned – skipping demo actions.")

    finally:

        # Always ensure the environment is properly shut down

        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":

    run_skeleton_task()

