# run_skeleton_task.py (Completed Version)

import re

import time

import numpy as np

from env import setup_environment, shutdown_environment

from skill_code import *                 # noqa: F401,F403  (we just import every predefined skill)

from video import (init_video_writers,

                   recording_step,

                   recording_get_observation)

from object_positions import get_object_positions

# ------------------------------------------------------------------

# -------------  UTILITY:  PDDL-PARSER FOR EXPLORATION --------------

# ------------------------------------------------------------------

def _pddl_predicates(pddl_txt: str):

    """Return the set of predicates explicitly declared in (:predicates …)."""

    content = re.findall(r'\(:predicates([\s\S]*?)\)', pddl_txt)

    if not content:

        return set()

    tokens = re.findall(r'\(([a-zA-Z0-9_\-]+)', content[0])

    return set(tokens)

def _pddl_all_tokens(pddl_txt: str):

    """Return *every* token that appears right after an opening parenthesis."""

    return set(re.findall(r'\(([a-zA-Z0-9_\-]+)', pddl_txt))

def find_missing_predicates(pddl_txt: str):

    """Return predicates that are *used* in actions but *not* declared."""

    declared = _pddl_predicates(pddl_txt)

    used = _pddl_all_tokens(pddl_txt)

    reserved = {

        'and', 'not', '=', 'forall', 'when',

        ':action', ':parameters', ':precondition', ':effect',

        ':types', ':predicates', ':requirements',

        'define', 'domain'

    }

    # remove every reserved keyword and already declared predicate

    missing = {tok for tok in used if tok not in reserved and tok not in declared}

    return missing

# ------------------------------------------------------------------

# -------------  EXPLORATION-DOMAIN (as given in prompt) ------------

# ------------------------------------------------------------------

EXPLORATION_DOMAIN_TXT = r"""

(define (domain exploration)

  (:requirements :strips :typing :conditional-effects :universal-preconditions)

  (:types 

    robot object location

  )

  (:predicates

    (robot-at ?r - robot ?loc - location)

    (at ?obj - object ?loc - location)

    (identified ?obj - object)

    (temperature-known ?obj - object)

    (holding ?obj - object)

    (handempty)

    (weight-known ?obj - object)

    (durability-known ?obj - object)

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (identified ?obj)

        )

      )

    )

  )

  (:action move

    :parameters (?r - robot ?from - location ?to - location)

    :precondition (robot-at ?r ?from)

    :effect (and

      (not (robot-at ?r ?from))

      (robot-at ?r ?to)

      (forall (?obj - object)

        (when (at ?obj ?to)

          (temperature-known ?obj)

        )

      )

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (weight-known ?obj)

    )

  )

  (:action pick

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (handempty)

    )

    :effect (and

      (holding ?obj)

      (not (handempty))

      (not (at ?obj ?loc))

      (durability-known ?obj)

    )

  )

  (:action pull

    :parameters (?r - robot ?obj - object ?loc - location)

    :precondition (and

       (robot-at ?r ?loc)

       (at ?obj ?loc)

       (holding ?obj)

       (not (lock-known ?obj))

    )

    :effect (lock-known ?obj)

  )

)

"""

# ------------------------------------------------------------------

# -------------  MAIN: EXECUTE SKELETON TASK ------------------------

# ------------------------------------------------------------------

def run_skeleton_task():

    """Generic skeleton for running any task in your simulation.

    In addition to the original skeleton, this implementation

    performs a lightweight 'exploration phase' that parses the given

    exploration-domain PDDL and prints out any predicates that are

    referenced in actions but *not* declared in the (:predicates …)

    section.  This is exactly how we discover that 'lock-known'

    is missing, as pointed out in the feedback.

    """

    print("\n===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------

    # 1)  Environment Setup

    # ------------------------------------------------------------------

    env, task = setup_environment()

    try:

        descriptions, obs = task.reset()

        # ------------- Optional Video Recording ----------------------

        init_video_writers(obs)

        task.step = recording_step(task.step)

        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------

        # 2)  PDDL-BASED EXPLORATION  – Find Missing Predicate

        # ------------------------------------------------------------------

        missing_predicates = find_missing_predicates(EXPLORATION_DOMAIN_TXT)

        if missing_predicates:

            print("\n[Exploration-Phase] Missing predicate(s) detected:")

            for pred in sorted(missing_predicates):

                print(f"  • {pred}")

        else:

            print("\n[Exploration-Phase] No missing predicates detected!")

        # ------------------------------------------------------------------

        # 3)  Object-List Consistency Check  (feedback: 'rubbish')

        # ------------------------------------------------------------------

        if hasattr(env, "get_object_list"):

            try:

                obj_list = env.get_object_list()

            except Exception as e:

                print(f"[Warning] env.get_object_list() failed: {e}")

                obj_list = []

        else:

            obj_list = []

        if 'rubbish' not in obj_list:

            print("\n[Consistency-Check] Object 'rubbish' is MISSING "

                  "from the environment object list!")

        else:

            print("\n[Consistency-Check] Object 'rubbish' found in the environment.")

        # ------------------------------------------------------------------

        # 4)  Minimal Skill-Invocation Demo

        #     (We must use only predefined skills; here we simply call

        #      rotate to align the gripper with a neutral orientation.)

        # ------------------------------------------------------------------

        print("\n[Demo] Performing a neutral gripper rotation ...")

        target_quat = np.array([0.0, 0.0, 0.0, 1.0], dtype=np.float32)

        try:

            _obs, _reward, _done = rotate(

                env,

                task,

                target_quat=target_quat,

                max_steps=50,

                threshold=0.02,

                timeout=5.0

            )

        except Exception as e:

            print(f"[Demo] rotate() failed: {e}")

        # ------------------------------------------------------------------

        # 5)  Placeholder for further task-specific actions

        # ------------------------------------------------------------------

        # NOTE:

        #   This skeleton purposely leaves out a concrete task plan because

        #   the assignment focuses on (a) finding the missing predicate and

        #   (b) demonstrating correct usage of predefined skills.

        #   Implement your full plan logic here if/when needed.

        # Small sleep so that logs can be read comfortably

        time.sleep(0.5)

    finally:

        # ------------------------------------------------------------------

        # 6)  Always shut down the environment properly

        # ------------------------------------------------------------------

        shutdown_environment(env)

    print("\n===== End of Skeleton Task =====\n")

if __name__ == "__main__":

    run_skeleton_task()

