# run_skeleton_task.py (Executable – completed)

import os
import re
import time
from typing import Set, List

import numpy as np
from pyrep.objects.shape import Shape            # kept – part of the skeleton
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *                          # DO NOT redefine existing skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper functions for the “exploration” phase (predicate scan)
# -----------------------------------------------------------

def _extract_predicates_from_section(section: str) -> Set[str]:
    """
    Very small-footprint PDDL tokenizer that returns the set of the first
    tokens inside parentheses – i.e. the predicate / function names that
    actually occur in a block of PDDL.
    """
    # Remove comments and linebreaks
    section = re.sub(r";.*", "", section)
    # Anything that looks like "(xxx"               -> xxx
    return set(re.findall(r"\(\s*([^\s\(\)]+)", section))


def find_missing_predicates(domain_str: str) -> List[str]:
    """
    Scans a PDDL domain description and returns every predicate that is used in
    preconditions / effects but never declared in the global (:predicates …)
    block.
    """
    # --- 1) locate the global (:predicates …) block --------------------------
    predicate_block = re.search(r"\(:predicates(.*?)\)", domain_str, re.S).group(1)
    declared_predicates = _extract_predicates_from_section(predicate_block)

    # --- 2) locate all action blocks ----------------------------------------
    action_blocks = re.findall(r"\(:action.*?\)", domain_str, re.S)

    used_predicates: Set[str] = set()
    for block in action_blocks:
        # we only want the predicates *inside* precondition / effect, so strip
        # the action header "( :action <name> …"
        body = re.sub(r"\(:action.*?:parameters.*?:precondition", "", block, flags=re.S)
        used_predicates.update(_extract_predicates_from_section(body))

    # --- 3) special keywords that should be ignored -------------------------
    pddl_keywords = {
        "and", "or", "not", "forall", "when", "exists", "=", ">", "<", "<=",
        ">=", "+", "-", "*", "/", "assign", "increase", "decrease"
    }
    used_predicates.difference_update(pddl_keywords)

    # ------------------------------------------------------------------------
    missing = sorted(list(used_predicates - declared_predicates))
    return missing


# ---------------------------------------------------------------------------
# The exploration domain string (shortened to the essentials from the prompt)
# ---------------------------------------------------------------------------
EXPLORATION_DOMAIN = """
(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types robot object location)

  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
  )

  (:action move
    :parameters (?r - robot ?from - location ?to - location)
    :precondition (robot-at ?r ?from)
    :effect (and
      (not (robot-at ?r ?from))
      (robot-at ?r ?to)
      (forall (?obj - object)
        (when (at ?obj ?to)
          (identified ?obj)
        )
      )
    )
  )

  (:action pull
    :parameters (?r - robot ?obj - object ?loc - location)
    :precondition (and
       (robot-at ?r ?loc)
       (at ?obj ?loc)
       (holding ?obj)
       (not (lock-known ?obj))
    )
    :effect (lock-known ?obj)
  )
)
"""


# ============================= Main runner ==================================

def run_skeleton_task():
    """Generic skeleton extended with a minimal ‘exploration’ phase that
    detects missing predicates, prints them, then runs a small demonstration
    loop (if possible) before shutting the environment down gracefully."""
    print("===== Starting Skeleton Task =====")

    # === 0) Exploration phase – purely symbolic ============================
    print("\n[Exploration] Scanning exploration domain for undeclared predicates…")
    missing_preds = find_missing_predicates(EXPLORATION_DOMAIN)
    if missing_preds:
        print(f"[Exploration] Missing predicate(s) detected: {missing_preds}")
    else:
        print("[Exploration] No missing predicates found.")

    # === 1) Environment Setup =============================================
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap the step + get_observation calls for optional video recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === 2) Retrieve Object Positions ==================================
        positions = get_object_positions()
        if not positions:
            print("[Task] WARNING: 'object_positions' returned no data. "
                  "Skipping demonstration skills.")
        else:
            # Pick an arbitrary object & a target – purely demonstrational
            object_name, object_pos = next(iter(positions.items()))
            print(f"[Task] Demonstration: will attempt to 'pick' object "
                  f"'{object_name}' at {np.round(object_pos, 3)}")

            # Configurable parameters (feedback-driven improvements)
            cfg = {
                "approach_distance": float(os.getenv("APPROACH_DISTANCE", 0.15)),
                "max_steps":         int(float(os.getenv("MAX_STEPS", 125))),
                "threshold":         float(os.getenv("WP_THRESHOLD", 0.01)),
                "approach_axis":     os.getenv("APPROACH_AXIS", "z"),
                "timeout":           float(os.getenv("SKILL_TIMEOUT", 10.0)),
            }

            # --- try ‘pick’ -------------------------------------------------
            try:
                obs, reward, done = pick(
                    env, task,
                    target_pos=np.array(object_pos),
                    approach_distance=cfg["approach_distance"],
                    max_steps=cfg["max_steps"],
                    threshold=cfg["threshold"],
                    approach_axis=cfg["approach_axis"],
                    timeout=cfg["timeout"]
                )
                if done:
                    print("[Task] Task finished during pick – early terminate.")
                    return
            except Exception as e:
                print(f"[Task] Pick attempt failed: {e}")

            # --- place back to the same position as a quick sanity check ---
            try:
                obs, reward, done = place(
                    env, task,
                    target_pos=np.array(object_pos),
                    approach_distance=cfg["approach_distance"],
                    max_steps=cfg["max_steps"],
                    threshold=cfg["threshold"],
                    approach_axis='-z',          # place from above by default
                    timeout=cfg["timeout"]
                )
            except Exception as e:
                print(f"[Task] Place attempt failed: {e}")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()