# run_skeleton_task.py (Completed Version – with Exploration Phase)

import re

import math

import time

import numpy as np

from pyrep.objects.shape import Shape

from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# all primitive / high-level skills come from this module

from skill_code import *          # noqa: F403,F401

from video import (

    init_video_writers,

    recording_step,

    recording_get_observation,

)

from object_positions import get_object_positions

# -------------------------------------------------------------------

#                       DOMAIN-PARSING UTILITIES

# -------------------------------------------------------------------

COMBINED_DOMAIN_STR = r"""

(define (domain combined-domain)

  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)

  (:types object location drawer - object gripper - object position - object angle - object)

  (:predicates

    (at ?obj - object ?loc - location)

    (holding ?obj - object)

    (handempty)

    (is-locked ?d - drawer)

    (is-open ?d - drawer)

    (rotated ?g - gripper ?a - angle)

    (gripper-at ?g - gripper ?p - position)

    (holding-drawer ?g - gripper ?d - drawer)

    (is-side-pos ?p - position ?d - drawer)

    (is-anchor-pos ?p - position ?d - drawer)

  )

)

"""

EXPLORATION_DOMAIN_STR = r"""

(define (domain exploration)

  (:requirements :strips :typing :conditional-effects :universal-preconditions)

  (:types robot object location)

  (:predicates

    (robot-at ?r - robot ?loc - location)

    (at ?obj - object ?loc - location)

    (identified ?obj - object)

    (temperature-known ?obj - object)

    (holding ?obj - object)

    (handempty)

    (weight-known ?obj - object)

    (durability-known ?obj - object)

  )

  (:action pull

      :parameters (?r - robot ?obj - object ?loc - location)

      :precondition (and

         (robot-at ?r ?loc)

         (at ?obj ?loc)

         (holding ?obj)

         (not (lock-known ?obj))
      )

      :effect (lock-known ?obj)

  )

)

"""

def _extract_declared_predicates(domain_str: str) -> set:

    """Return the set of predicates declared inside (:predicates …)."""

    declared = set()

    pattern = re.compile(r'\(:predicates([\s\S]*?)\)', re.MULTILINE)

    m = pattern.search(domain_str)

    if not m:

        return declared

    block = m.group(1)

    for token in re.findall(r'\([^\s]+', block):

        declared.add(token.strip('('))

    return declared

def _extract_used_predicates(domain_str: str) -> set:

    """Return every token starting with '(' that looks like a predicate."""

    used = set()

    for token in re.findall(r'\(([^\s]+)', domain_str):

        # skip PDDL keywords

        if token.startswith(':'):

            continue

        if token in {

            'and',

            'not',

            '=',

            'when',

            'forall',

        }:

            continue

        used.add(token)

    return used

def identify_missing_predicates() -> set:

    """Compare domains and locate missing predicates that appear in operators

    but are *not* declared."""

    declared = _extract_declared_predicates(COMBINED_DOMAIN_STR) | _extract_declared_predicates(

        EXPLORATION_DOMAIN_STR

    )

    used = _extract_used_predicates(EXPLORATION_DOMAIN_STR)

    missing = used - declared

    if missing:

        print("\n[Exploration]  ⚠  Detected predicates referenced but not declared:", missing, "\n")

    else:

        print("\n[Exploration]  ✓  No missing predicates detected.\n")

    return missing

# -------------------------------------------------------------------

#               SMALL MATH/GEOMETRY HELPERS (Quaternion)

# -------------------------------------------------------------------

def quaternion_from_euler(roll: float, pitch: float, yaw: float):

    """Convert Euler XYZ (rad) to quaternion (xyzw)."""

    cr = math.cos(roll / 2)

    sr = math.sin(roll / 2)

    cp = math.cos(pitch / 2)

    sp = math.sin(pitch / 2)

    cy = math.cos(yaw / 2)

    sy = math.sin(yaw / 2)

    qx = sr * cp * cy - cr * sp * sy

    qy = cr * sp * cy + sr * cp * sy

    qz = cr * cp * sy - sr * sp * cy

    qw = cr * cp * cy + sr * sp * sy

    return np.array([qx, qy, qz, qw], dtype=np.float32)

# -------------------------------------------------------------------

#                       MAIN EXECUTION ROUTINE

# -------------------------------------------------------------------

def run_skeleton_task():

    """Generic skeleton for running any task in your simulation."""

    print("===== Starting Skeleton Task =====")

    # ----------------------------------------------------------------

    # 0) OFF-LINE EXPLORATION OF PDDL DOMAINS – detect missing pieces

    # ----------------------------------------------------------------

    missing_preds = identify_missing_predicates()

    # (Here we only *detect*.  If downstream logic needs the knowledge, we

    #  could adapt; for now we simply report.)

    # ----------------------------------------------------------------

    # 1) ENVIRONMENT INITIALISATION

    # ----------------------------------------------------------------

    env, task = setup_environment()

    try:

        # Reset the task to its initial state

        descriptions, obs = task.reset()

        print("[Env] Task descriptions:", descriptions)

        # (Optional) video writer setup

        init_video_writers(obs)

        # Wrap task.step / get_observation for video recording

        task.step = recording_step(task.step)

        task.get_observation = recording_get_observation(task.get_observation)

        # ----------------------------------------------------------------

        # 2) FETCH / INSPECT OBJECT POSITIONS

        # ----------------------------------------------------------------

        positions = get_object_positions()  # expected dict {name: (x, y, z)}

        if not positions:

            print("[Warning] object_positions returned an empty dictionary.")

        # crude discovery of drawers/handles

        drawer_handles = [

            (name, pos) for name, pos in positions.items() if "drawer" in name and "handle" in name

        ]

        movable_objects = [

            (name, pos)

            for name, pos in positions.items()

            if ("object" in name or "item" in name or "cube" in name)

        ]

        print(f"[Env] Detected {len(drawer_handles)} potential drawer handles.")

        print(f"[Env] Detected {len(movable_objects)} generic movable objects.")

        # ----------------------------------------------------------------

        # 3) DEMO PLAN – open first drawer (if detected)

        # ----------------------------------------------------------------

        if drawer_handles:

            handle_name, handle_pos = drawer_handles[0]

            print(f"[Task] Attempting to open drawer via handle '{handle_name}'")

            # 3-a) Ensure gripper orientation is 90° wrt drawer – we assume rotation around Z

            current_quat = obs.gripper_pose[3:7]

            roll, pitch, yaw = euler_from_quat(current_quat)  # provided by skill_code

            target_quat = quaternion_from_euler(roll, pitch, yaw + math.pi / 2.0)

            try:

                print("[Skill] rotate → aligning gripper for side grasp.")

                obs, reward, done = rotate(  # noqa: F405  (imported from skill_code)

                    env,

                    task,

                    target_quat=target_quat,

                    max_steps=100,

                    threshold=0.05,

                    timeout=10.0,

                )

                if done:

                    print("[Task] Episode ended during rotate.")

                    return

            except Exception as exc:

                print("[Error] rotate failed:", exc)

            # 3-b) Approach drawer handle & pick

            try:

                print("[Skill] move → approaching handle.")

                obs, reward, done = move(  # noqa: F405

                    env,

                    task,

                    target_pos=handle_pos,

                    approach_distance=0.15,

                    max_steps=120,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=15.0,

                )

                if done:

                    print("[Task] Episode ended during move.")

                    return

            except Exception as exc:

                print("[Error] move failed:", exc)

            try:

                print("[Skill] pick → grasping handle.")

                obs, reward, done = pick(  # noqa: F405

                    env,

                    task,

                    target_pos=handle_pos,

                    approach_distance=0.08,

                    max_steps=80,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=10.0,

                )

                if done:

                    print("[Task] Episode ended during pick.")

                    return

            except Exception as exc:

                print("[Error] pick failed:", exc)

            # 3-c) Pull the drawer open

            try:

                print("[Skill] pull → pulling drawer outward.")

                obs, reward, done = pull(  # noqa: F405

                    env,

                    task,

                    pull_distance=0.20,

                    max_steps=120,

                    threshold=0.01,

                    timeout=15.0,

                )

                if done:

                    print("[Task] Episode ended during pull.")

                    return

            except Exception as exc:

                print("[Error] pull failed:", exc)

            # 3-d) Release handle

            try:

                print("[Skill] place → releasing handle.")

                current_gripper_pos = task.get_observation().gripper_pose[:3]

                obs, reward, done = place(  # noqa: F405

                    env,

                    task,

                    target_pos=current_gripper_pos,

                    approach_distance=0.04,

                    max_steps=60,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=10.0,

                )

                if done:

                    print("[Task] Episode ended during place.")

                    return

            except Exception as exc:

                print("[Error] place failed:", exc)

        # ----------------------------------------------------------------

        # 4) OPTIONAL: manipulate a small object (if any)

        # ----------------------------------------------------------------

        if movable_objects:

            obj_name, obj_pos = movable_objects[0]

            print(f"[Task] Demonstration pick-and-place for object '{obj_name}'.")

            # pick object

            try:

                print("[Skill] move → approach object.")

                obs, reward, done = move(

                    env,

                    task,

                    target_pos=obj_pos,

                    approach_distance=0.15,

                    max_steps=120,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=15.0,

                )

                if done:

                    return

                print("[Skill] pick → grasp object.")

                obs, reward, done = pick(

                    env,

                    task,

                    target_pos=obj_pos,

                    approach_distance=0.06,

                    max_steps=80,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=10.0,

                )

                if done:

                    return

            except Exception as exc:

                print("[Error] Failed to pick movable object:", exc)

            # place object inside opened drawer (approximate – assume +X direction)

            try:

                drop_target = np.array(obj_pos) + np.array([0.25, 0.0, 0.0])

                print("[Skill] move → to drop location inside drawer.")

                obs, reward, done = move(

                    env,

                    task,

                    target_pos=drop_target,

                    approach_distance=0.12,

                    max_steps=120,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=15.0,

                )

                if done:

                    return

                print("[Skill] place → drop object.")

                obs, reward, done = place(

                    env,

                    task,

                    target_pos=drop_target,

                    approach_distance=0.05,

                    max_steps=80,

                    threshold=0.01,

                    approach_axis="z",

                    timeout=10.0,

                )

            except Exception as exc:

                print("[Error] Failed to place movable object:", exc)

        # ----------------------------------------------------------------

        # 5) END OF DEMO / PLAN

        # ----------------------------------------------------------------

        print("[Task] Finished demo plan.")

    finally:

        # always cleanly terminate simulation

        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":

    run_skeleton_task()
