import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
import traceback

from env import setup_environment, shutdown_environment
from skill_code import *          # noqa: F401,F403  (import all predefined skills)
from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)
from object_positions import get_object_positions


def run_skeleton_task():
    """
    Execute the oracle plan (rotate → move → move → pick‑drawer → pull → pick 2 × tomatoes → place)
    using ONLY the predefined skills imported from `skill_code`.
    """
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()

    try:
        # Reset task
        descriptions, obs = task.reset()

        # ---------------- Video Recording ----------------
        init_video_writers(obs)
        task.step = recording_step(task.step)          # wrap step for video
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Retrieve all relevant object positions
        # ------------------------------------------------------------------
        positions = get_object_positions()     # dictionary: name -> np.ndarray([x,y,z])

        # Mandatory keys we expect from get_object_positions()
        required_keys = [
            "bottom_side_pos",
            "bottom_anchor_pos",
            "item1",
            "item2",
            "plate"
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[run] Missing key '{k}' in object positions. "
                               "Ensure get_object_positions() provides this entry.")

        # ------------------------------------------------------------------
        #  STEP‑BY‑STEP PLAN EXECUTION
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # -------- Step 1 : rotate gripper to 90 deg yaw --------
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("\n[Plan] Step 1 – Rotate gripper 90 deg (yaw)")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended unexpectedly after rotate.")
            return

        # -------- Step 2 : move to bottom_side_pos --------------
        print("\n[Plan] Step 2 – Move to bottom_side_pos:", positions["bottom_side_pos"])
        obs, reward, done = move(env, task, positions["bottom_side_pos"])
        if done:
            print("[Plan] Task ended unexpectedly after move to side position.")
            return

        # -------- Step 3 : move to bottom_anchor_pos ------------
        print("\n[Plan] Step 3 – Move to bottom_anchor_pos:", positions["bottom_anchor_pos"])
        obs, reward, done = move(env, task, positions["bottom_anchor_pos"])
        if done:
            print("[Plan] Task ended unexpectedly after move to anchor position.")
            return

        # -------- Step 4 : pick the drawer handle --------------
        print("\n[Plan] Step 4 – Pick drawer handle (bottom)")
        obs, reward, done = pick(env, task, positions["bottom_anchor_pos"],
                                 approach_distance=0.10, approach_axis='z')
        if done:
            print("[Plan] Task ended unexpectedly after picking drawer handle.")
            return

        # -------- Step 5 : pull drawer outward -----------------
        print("\n[Plan] Step 5 – Pull drawer outward")
        # Empirically choose 0.25 m along +x; adjust if environment differs
        obs, reward, done = pull(env, task, pull_distance=0.25, pull_axis='x')
        if done:
            print("[Plan] Task ended unexpectedly after pulling drawer.")
            return

        # -------- Step 6 : pick tomato1 ------------------------
        print("\n[Plan] Step 6 – Pick tomato1 (item1)")
        obs, reward, done = pick(env, task, positions["item1"],
                                 approach_distance=0.12, approach_axis='z')
        if done:
            print("[Plan] Task ended unexpectedly after picking tomato1.")
            return

        # -------- Step 7 : place tomato1 on plate --------------
        print("\n[Plan] Step 7 – Place tomato1 on plate")
        obs, reward, done = place(env, task, positions["plate"],
                                  approach_distance=0.12, approach_axis='z')
        if done:
            print("[Plan] Task ended unexpectedly after placing tomato1.")
            return

        # -------- Step 8 : pick tomato2 ------------------------
        print("\n[Plan] Step 8 – Pick tomato2 (item2)")
        obs, reward, done = pick(env, task, positions["item2"],
                                 approach_distance=0.12, approach_axis='z')
        if done:
            print("[Plan] Task ended unexpectedly after picking tomato2.")
            return

        # -------- Step 9 : place tomato2 on plate --------------
        print("\n[Plan] Step 9 – Place tomato2 on plate")
        obs, reward, done = place(env, task, positions["plate"],
                                  approach_distance=0.12, approach_axis='z')
        # Task completed!
        if done:
            print("\n[Plan] Task finished successfully! Reward:", reward)
        else:
            print("\n[Plan] Task sequence executed. (Environment has not returned done=True)")

    except Exception as e:
        print("=== Exception during task execution ===")
        traceback.print_exc()
    finally:
        # ------------------------------------------------------------------
        #  Shutdown environment – always called
        # ------------------------------------------------------------------
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
