# run_skeleton_task.py – Completed implementation
#
# Description:
#  - Executes the oracle plan to (1) open the bottom drawer, then (2) move two
#    tomatoes onto the plate while leaving other items untouched.
#  - Uses only the predefined skills provided in skill_code.

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Pre‑defined skills (DO NOT redefine) ===
from skill_code import rotate, move, pull, pick, place


def run_skeleton_task():
    """Run the task following the oracle plan."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    # 1) Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # --- (Optional) video capture helpers -------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2) Retrieve (x, y, z) positions for all relevant objects
        # ------------------------------------------------------------------
        # Primary source – helper module
        positions = get_object_positions()

        # Convenience accessor – falls back to direct Shape query
        def get_pos(name):
            if name in positions:
                return np.array(positions[name], dtype=float)
            # Fallback: query the simulator directly
            try:
                return np.array(Shape(name).get_position(), dtype=float)
            except Exception as exc:
                raise RuntimeError(f"[Error] Could not resolve position for '{name}': {exc}")

        # ------------------------------------------------------------------
        # 3) Oracle plan execution
        # ------------------------------------------------------------------
        done = False

        # ---------------- Step 1 – rotate gripper 90 deg -----------------
        #   (rotate from current quaternion around Z by +90°)
        cur_quat = obs.gripper_pose[3:7]
        eul = R.from_quat(cur_quat).as_euler('xyz')
        eul[2] += np.deg2rad(90.0)             # +90 deg yaw
        tgt_quat = R.from_euler('xyz', eul).as_quat()
        obs, reward, done = rotate(env, task, tgt_quat)
        if done:
            print("[Plan] Finished early after rotate."); return

        # ---------------- Step 2 – move to side‑pos of bottom drawer -----
        obs, reward, done = move(env, task, get_pos('bottom_side_pos'))
        if done:
            print("[Plan] Finished early after move to side."); return

        # ---------------- Step 3 – move to anchor‑pos of bottom drawer ---
        obs, reward, done = move(env, task, get_pos('bottom_anchor_pos'))
        if done:
            print("[Plan] Finished early after move to anchor."); return

        # ---------------- Step 4 – pull the drawer open ------------------
        #   Pull 15 cm along +X (RLBench open‑drawer tasks usually open along +X)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Plan] Finished early after pull."); return

        # ---------------- Steps 5‑12 – move two tomatoes to plate --------
        #   The spec mentions ‘tomato1’ and ‘tomato2’ but the scene
        #   contains ‘item1’/‘item2’.  Provide a mapping with graceful
        #   fallback to handle either naming convention.
        tomato_map = {
            'tomato1': 'tomato1' if 'tomato1' in positions else 'item1',
            'tomato2': 'tomato2' if 'tomato2' in positions else 'item2',
        }

        for tomato_alias in ('tomato1', 'tomato2'):
            real_name = tomato_map[tomato_alias]

            # ---------- Step X: move above the tomato -------------------
            obs, reward, done = move(env, task, get_pos(real_name))
            if done:
                print(f"[Plan] Finished early while approaching {real_name}."); return

            # ---------- Step X: pick the tomato -------------------------
            obs, reward, done = pick(env, task, get_pos(real_name))
            if done:
                print(f"[Plan] Finished early while picking {real_name}."); return

            # ---------- Step X: move to plate ---------------------------
            obs, reward, done = move(env, task, get_pos('plate'))
            if done:
                print(f"[Plan] Finished early while moving {real_name} to plate."); return

            # ---------- Step X: place the tomato on the plate -----------
            obs, reward, done = place(env, task, get_pos('plate'))
            if done:
                print(f"[Plan] Finished early while placing {real_name}."); return

        # ------------------------------------------------------------------
        # 4) Task completed
        # ------------------------------------------------------------------
        print("===== Task completed successfully! =====")

    finally:
        # Always shut down the environment to free resources
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()