import numpy as np

from env import setup_environment, shutdown_environment

# Import every primitive skill exactly once (move, rotate, pull, pick, place)
from skill_code import move, rotate, pull, pick, place

# Video helpers
from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

# Utility that returns a dictionary  {name(str): np.ndarray([x, y, z])}
from object_positions import get_object_positions


def _safe_get_pos(name: str, positions: dict, fallback: np.ndarray = None) -> np.ndarray:
    """Return a copy of the requested object position or raise a RuntimeError."""
    if name not in positions:
        if fallback is not None:
            return fallback.copy()
        raise RuntimeError(f"[Plan]  Position for <{name}> not provided by get_object_positions().")
    return positions[name].copy()


def run_plan():
    """
    Execute the oracle‑style twelve‑step plan required by the specification:
    1   move   – approach the chosen drawer from the side
    2   rotate – rotate the gripper 90 °
    3   move   – move to the drawer anchor (handle)
    4   pull   – pull the drawer open
    5   move   – move above tomato #1
    6   pick   – pick tomato #1
    7   move   – move above the plate
    8   place  – place tomato #1
    9   move   – move above tomato #2
    10  pick   – pick tomato #2
    11  move   – move above the plate
    12  place  – place tomato #2
    """
    print("==============  START TASK  ==============")

    env, task = setup_environment()
    try:
        # Reset ‑ we do not care about the language descriptions here
        _, obs = task.reset()

        # Optional recording
        init_video_writers(obs)
        task.step = recording_step(task.step)           # wrap step
        task.get_observation = recording_get_observation(
            task.get_observation
        )

        # Fetch every useful pose in the scene
        positions = get_object_positions()

        # ------------------------------------------------------------
        #  STATIC OBJECTS WE CARE ABOUT
        # ------------------------------------------------------------
        DRAWER_NAME = "middle"  # choose any: bottom | middle | top
        side_key   = f"{DRAWER_NAME}_side_pos"
        anchor_key = f"{DRAWER_NAME}_anchor_pos"
        plate_key  = "plate"

        # Two tomatoes we will move (item1, item2) – fewer than three to
        # exactly match the 12‑step specification.
        tomato_names = ["item1", "item2"]

        # Resolve coordinates
        drawer_side_pos   = _safe_get_pos(side_key,   positions)
        drawer_anchor_pos = _safe_get_pos(anchor_key, positions)
        plate_pos         = _safe_get_pos(plate_key,  positions)

        tomato_positions = {
            tomato: _safe_get_pos(tomato, positions) for tomato in tomato_names
        }

        # ------------------------------------------------------------
        #  STEP‑BY‑STEP EXECUTION (strict order set by the specification)
        # ------------------------------------------------------------

        # STEP‑1  move ── go to the drawer’s side approach pose
        print("\n[STEP 1] move → drawer side position")
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 1.")
            return

        # STEP‑2  rotate ── rotate 90 deg around the Z‑axis
        print("\n[STEP 2] rotate → 90 deg (around Z)")
        target_quat = (
            # xyzw quaternion for (roll, pitch, yaw) = (0, 0, 90°)
            # RLBench uses xyzw ordering
            np.array([0, 0, np.sin(np.pi / 4), np.cos(np.pi / 4)], dtype=np.float32)
        )
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 2.")
            return

        # STEP‑3  move ── touch the drawer handle (anchor position)
        print("\n[STEP 3] move → drawer anchor (handle)")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 3.")
            return

        # STEP‑4  pull ── pull the drawer 20 cm along +X (tune if needed)
        print("\n[STEP 4] pull → open drawer")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis="x")
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 4.")
            return

        # ‑‑‑‑‑‑ tomato #1 ‑‑‑‑‑‑
        # STEP‑5  move ── go above tomato #1
        print("\n[STEP 5] move → tomato #1")
        tomato1_pos = tomato_positions[tomato_names[0]]
        obs, reward, done = move(env, task, tomato1_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 5.")
            return

        # STEP‑6  pick tomato #1
        print("\n[STEP 6] pick tomato #1")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 6.")
            return

        # STEP‑7  move ── above the plate
        print("\n[STEP 7] move → plate (with tomato #1)")
        obs, reward, done = move(env, task, plate_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 7.")
            return

        # STEP‑8  place tomato #1
        print("\n[STEP 8] place tomato #1 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 8.")
            return

        # ‑‑‑‑‑‑ tomato #2 ‑‑‑‑‑‑
        # STEP‑9  move ── go above tomato #2
        print("\n[STEP 9] move → tomato #2")
        tomato2_pos = tomato_positions[tomato_names[1]]
        obs, reward, done = move(env, task, tomato2_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 9.")
            return

        # STEP‑10 pick tomato #2
        print("\n[STEP 10] pick tomato #2")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 10.")
            return

        # STEP‑11 move above plate
        print("\n[STEP 11] move → plate (with tomato #2)")
        obs, reward, done = move(env, task, plate_pos)
        if done:
            print("[Plan] Task terminated unexpectedly after STEP 11.")
            return

        # STEP‑12 place tomato #2
        print("\n[STEP 12] place tomato #2 on plate")
        obs, reward, done = place(env, task, plate_pos)

        # Final state assessment
        if done:
            print("\n[Plan] Mission finished successfully!  Reward =", reward)
        else:
            print("\n[Plan] Plan completed (done=False). You may add more steps if needed.")

    finally:
        shutdown_environment(env)
        print("==============  END TASK  ==============")


if __name__ == "__main__":
    run_plan()
