# run_oracle_plan.py
#
# This script implements the concrete oracle plan required for the task
# “Unlock the cabinet by opening an accessible drawer, then transfer the
# 2 tomatoes onto the plate, leaving any other items undisturbed.”
#
# Only the skills defined in skill_code are used, in the exact order
# prescribed by the specification (rotate → move → move → pull → …).

import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# Import every predefined skill exactly as‑is
from skill_code import rotate, move, pull, pick, place


def run_oracle_plan():
    print("\n================  START ORACLE PLAN  ================\n")

    # --------------------------------------------------
    #  Environment set‑up
    # --------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the task – grabs the initial observation
        descriptions, obs = task.reset()

        # Initialise (optional) video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------------------------------------------
        #  Cache all useful 3‑D positions
        # --------------------------------------------------
        positions = get_object_positions()
        # Drawer positions
        drawer_side  = positions['bottom_side_pos']     # reachable “side” pose
        drawer_anchor = positions['bottom_anchor_pos']  # “anchor” pose for pulling

        # Items & plate
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        plate_pos   = positions['plate']

        # A tiny lift to avoid collisions when moving directly above objects
        Z_LIFT = 0.15

        # --------------------------------------------------
        #  Specification steps 1‑12
        # --------------------------------------------------

        # STEP‑1  rotate
        #   – Rotate the wrist 90° about the Z‑axis (world frame)
        target_quat = R.from_euler('xyz', [0, 0, np.deg2rad(90)]).as_quat()
        print("\n[STEP‑1] rotate → 90° about Z\n")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 1!")
            return

        # STEP‑2  move  → drawer‑side
        print("\n[STEP‑2] move → drawer side position\n")
        side_above = drawer_side + np.array([0, 0, Z_LIFT])
        obs, reward, done = move(env, task, side_above)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 2!")
            return

        # STEP‑3  move  → drawer‑anchor (handle)
        print("\n[STEP‑3] move → drawer anchor (handle) position\n")
        obs, reward, done = move(env, task, drawer_anchor)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 3!")
            return

        # STEP‑4  pull  → open the drawer
        print("\n[STEP‑4] pull → pull drawer outward\n")
        # Pull 20 cm along +X (assumes drawer opens along +X)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 4!")
            return

        # --------------------------------------------------
        #  Move tomato‑1 onto plate
        # --------------------------------------------------

        # STEP‑5  move  → above tomato‑1
        print("\n[STEP‑5] move → above tomato‑1\n")
        obs, reward, done = move(env, task, tomato1_pos + np.array([0, 0, Z_LIFT]))
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 5!")
            return

        # STEP‑6  pick  tomato‑1
        print("\n[STEP‑6] pick → tomato‑1\n")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 6!")
            return

        # STEP‑7  move  → above plate
        print("\n[STEP‑7] move → above plate\n")
        obs, reward, done = move(env, task, plate_pos + np.array([0, 0, Z_LIFT]))
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 7!")
            return

        # STEP‑8  place tomato‑1 on plate
        print("\n[STEP‑8] place → tomato‑1 on plate\n")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 8!")
            return

        # --------------------------------------------------
        #  Move tomato‑2 onto plate
        # --------------------------------------------------

        # STEP‑9  move  → above tomato‑2
        print("\n[STEP‑9] move → above tomato‑2\n")
        obs, reward, done = move(env, task, tomato2_pos + np.array([0, 0, Z_LIFT]))
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 9!")
            return

        # STEP‑10 pick  tomato‑2
        print("\n[STEP‑10] pick → tomato‑2\n")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 10!")
            return

        # STEP‑11 move  → above plate
        print("\n[STEP‑11] move → above plate\n")
        obs, reward, done = move(env, task, plate_pos + np.array([0, 0, Z_LIFT]))
        if done:
            print("[Early‑Exit] Task finished unexpectedly at step 11!")
            return

        # STEP‑12 place tomato‑2 on plate
        print("\n[STEP‑12] place → tomato‑2 on plate\n")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Completion] Task concluded at step 12. Reward:", reward)
        else:
            print("[Completion] All planned steps executed. Task may continue running…")

    finally:
        shutdown_environment(env)
        print("\n================   END ORACLE PLAN   ================\n")


if __name__ == "__main__":
    run_oracle_plan()
