import numpy as np
from scipy.spatial.transform import Rotation as R

# === RLBench / Helper Imports ===
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === All low‑level skills are already defined in the external skill_code module ===
# They are imported here via wildcard so we can call them directly
from skill_code import *

# ------------------------------
# Main routine that executes the oracle plan given in the specification
# ------------------------------
def run_skeleton_task():
    print("===== [Task] Pull drawer + place tomatoes =====")

    # ------------------------------------------------------------------
    # 1)  Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset environment and wrap it for optional video‑logging
        descriptions, obs = task.reset()
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------
        # 2)  Query all object positions we will need
        # ------------------------------------------------------------------
        pos = get_object_positions()  # dictionary: name ➜ np.ndarray([x,y,z])

        req_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'tomato1', 'tomato2', 'plate'
        ]
        for k in req_keys:
            if k not in pos:
                raise KeyError(f"[Task] Object position for '{k}' not found!")

        # Convenience aliases
        side_pos   = pos['bottom_side_pos']
        anchor_pos = pos['bottom_anchor_pos']
        tomato1_p  = pos['tomato1']
        tomato2_p  = pos['tomato2']
        plate_p    = pos['plate']

        # ------------------------------------------------------------------
        # 3)  Execute the oracle plan step‑by‑step
        # ------------------------------------------------------------------

        # ---- Step‑1 : rotate gripper to 90° about Z (open pose) ----
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # [x,y,z,w]
        print("\n[Plan] Step‑1  rotate gripper → 90°")
        obs, reward, done = rotate(env, task, quat_90deg)
        if done: return

        # ---- Step‑2 : move to drawer side position ----
        print("\n[Plan] Step‑2  move → bottom_side_pos")
        obs, reward, done = move(env, task, side_pos)
        if done: return

        # ---- Step‑3 : move to drawer anchor (handle) ----
        print("\n[Plan] Step‑3  move → bottom_anchor_pos")
        obs, reward, done = move(env, task, anchor_pos)
        if done: return

        # ---- Step‑4 : pick drawer handle (grasp) ----
        print("\n[Plan] Step‑4  grasp drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,  # small because we are already close
            approach_axis='-z'
        )
        if done: return

        # ---- Step‑5 : pull the drawer outwards ----
        print("\n[Plan] Step‑5  pull drawer (along +x, 0.20 m)")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.20,
            pull_axis='x'
        )
        if done: return

        # ---- Step‑6 : pick tomato‑1 ----
        print("\n[Plan] Step‑6  pick tomato1")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato1_p,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done: return

        # ---- Step‑7 : place tomato‑1 on plate ----
        print("\n[Plan] Step‑7  place tomato1 on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_p,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done: return

        # ---- Step‑8 : pick tomato‑2 ----
        print("\n[Plan] Step‑8  pick tomato2")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato2_p,
            approach_distance=0.15,
            approach_axis='z'
        )
        if done: return

        # ---- Step‑9 : place tomato‑2 on plate ----
        print("\n[Plan] Step‑9  place tomato2 on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_p,
            approach_distance=0.15,
            approach_axis='z'
        )

        # ------------------------------------------------------------------
        # 4)  Final status
        # ------------------------------------------------------------------
        if done:
            print("\n[Task] Episode finished naturally (done=True). Final reward:", reward)
        else:
            print("\n[Task] All plan steps executed. Environment did not signal done → assume success.")

    finally:
        shutdown_environment(env)
        print("===== [Task] Shutdown complete =====")


if __name__ == "__main__":
    run_skeleton_task()
