# run_combined_task.py

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# All low‑level skills are provided in the external module “skill_code”
from skill_code import rotate, move, pull, pick, place

from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

from object_positions import get_object_positions


def _safe_lookup(name: str, positions: dict):
    """Return position of an object name or raise a descriptive error."""
    if name not in positions or positions[name] is None:
        raise RuntimeError(f"[Task] Cannot find position for object: {name}")
    return positions[name]


def run_combined_task():
    """
    Execute the oracle plan (drawer‑opening + tomato placement) using only the
    predefined skills: rotate, move, pull, pick, place.
    """
    print("===== Starting Combined Task =====")

    env, task = setup_environment()
    try:
        # --- Reset task & wrap video recording ----------------------------
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --- Fetch all useful positions -----------------------------------
        positions = get_object_positions()
        # Drawer related positions
        middle_side_pos = _safe_lookup("middle_side_pos", positions)
        middle_anchor_pos = _safe_lookup("middle_anchor_pos", positions)
        # Tomatoes & plate
        tomato1_pos = _safe_lookup("tomato1", positions)
        tomato2_pos = _safe_lookup("tomato2", positions)
        plate_pos = _safe_lookup("plate", positions)

        # === Step‑by‑step execution of the oracle plan ====================

        # Step 1 : rotate gripper from zero_deg to ninety_deg
        target_quat = R.from_euler("z", 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished during rotate.")
            return

        # Step 2 : move‑to‑side (generic ‘move’ skill)
        obs, reward, done = move(env, task, middle_side_pos)
        if done:
            print("[Task] Finished during move‑to‑side.")
            return

        # Step 3 : move‑to‑anchor
        obs, reward, done = move(env, task, middle_anchor_pos)
        if done:
            print("[Task] Finished during move‑to‑anchor.")
            return

        # Step 4 : pick‑drawer   (map to generic ‘pick’ on drawer handle)
        obs, reward, done = pick(env, task, middle_anchor_pos, approach_distance=0.08)
        if done:
            print("[Task] Finished during pick‑drawer.")
            return

        # Step 5 : pull the drawer outwards (positive X axis, 0.20 m)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis="x")
        if done:
            print("[Task] Finished during pull.")
            return

        # Step 6 : pick tomato1 from the table
        obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15)
        if done:
            print("[Task] Finished while picking tomato1.")
            return

        # Step 7 : place tomato1 on the plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15)
        if done:
            print("[Task] Finished after placing tomato1.")
            return

        # Step 8 : pick tomato2 from the table
        obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15)
        if done:
            print("[Task] Finished while picking tomato2.")
            return

        # Step 9 : place tomato2 on the plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15)
        if done:
            print("[Task] Task successfully completed! Reward:", reward)
        else:
            print("[Task] Plan executed, but environment not marked as done.")

    finally:
        shutdown_environment(env)
        print("===== End of Combined Task =====")


if __name__ == "__main__":
    run_combined_task()