import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment

# Import every predefined low‑level skill exactly once
from skill_code import move, pick, place, rotate, pull

# Utilities for video recording (they are no‑ops if the module is absent)
from video import init_video_writers, recording_step, recording_get_observation

# Helper that returns a dict {object_name: np.ndarray([x, y, z])}
from object_positions import get_object_positions


def run_oracle_plan():
    """Execute the fixed 9‑step oracle plan described in the specification."""
    print("=====  Start Oracle Plan Execution  =====")

    # ─────────────────────────────────────────────────────────
    # 1)  Environment initialisation
    # ─────────────────────────────────────────────────────────
    env, task = setup_environment()          # RLBench helper
    try:
        descriptions, obs = task.reset()     # reset to initial state

        # optional video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)                       # wrap step
        task.get_observation = recording_get_observation(task.get_observation)

        # ─────────────────────────────────────────────────────
        # 2)  Cache every object (x, y, z) position we need
        # ─────────────────────────────────────────────────────
        positions = get_object_positions()
        try:
            side_pos   = positions['bottom_side_pos']
            anchor_pos = positions['bottom_anchor_pos']
            tomato1_pos = positions['tomato1']
            tomato2_pos = positions['tomato2']
            plate_pos   = positions['plate']
        except KeyError as ke:
            raise RuntimeError(f"Required object missing in get_object_positions(): {ke}")

        # ─────────────────────────────────────────────────────
        # 3)  Execute oracle plan step‑by‑step
        # ─────────────────────────────────────────────────────

        done = False  # RLBench termination flag

        # Step‑1  rotate(gripper, zero_deg, ninety_deg)
        target_quat = R.from_euler('z', 90.0, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Finished after rotate – task signalled completion.")
            return

        # Step‑2  move‑to‑side (go to drawer side position)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Finished after move‑to‑side.")
            return

        # Step‑3  move‑to‑anchor (drawer anchor position)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Finished after move‑to‑anchor.")
            return

        # Step‑4  pick‑drawer  (close gripper on the handle)
        obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.05)
        if done:
            print("[Plan] Finished after pick‑drawer.")
            return

        # Step‑5  pull (open drawer along +x)
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Plan] Finished after pull.")
            return

        # Step‑6  pick tomato1
        obs, reward, done = pick(env, task, tomato1_pos, approach_axis='z')
        if done:
            print("[Plan] Finished after picking tomato1.")
            return

        # Step‑7  place tomato1 on plate
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Plan] Finished after placing tomato1.")
            return

        # Step‑8  pick tomato2
        obs, reward, done = pick(env, task, tomato2_pos, approach_axis='z')
        if done:
            print("[Plan] Finished after picking tomato2.")
            return

        # Step‑9  place tomato2 on plate
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Plan] Finished after placing tomato2.")
        else:
            print("[Plan] Oracle plan executed – task did not signal completion (done=False).")

    finally:
        shutdown_environment(env)
        print("=====  Oracle Plan Execution Finished  =====")


if __name__ == "__main__":
    run_oracle_plan()