import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Bring all predefined skills into scope (move, pick, place, rotate, pull)
from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

# Utility that returns a dict with object names as keys and 3‑D world positions as values
from object_positions import get_object_positions


def run_skeleton_task():
    """Run the task: 1) open the bottom drawer, 2) place two tomatoes on the plate."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # --------------------------------------------------------------
        #  (Optional) video recording helpers
        # --------------------------------------------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)            # wrap step
        task.get_observation = recording_get_observation(
            task.get_observation
        )

        # --------------------------------------------------------------
        #  Retrieve positions of all relevant objects
        # --------------------------------------------------------------
        positions = get_object_positions()

        required_keys = [
            "bottom_side_pos",
            "bottom_anchor_pos",
            "item1",
            "item2",
            "plate",
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(
                    f"[Task] Required key '{k}' not found in object_positions!"
                )

        bottom_side_pos = np.asarray(positions["bottom_side_pos"])
        bottom_anchor_pos = np.asarray(positions["bottom_anchor_pos"])
        tomato1_pos = np.asarray(positions["item1"])
        tomato2_pos = np.asarray(positions["item2"])
        plate_pos = np.asarray(positions["plate"])

        # ------------------------------------------------------------------
        #  ORACLE PLAN (13 steps, exactly matching the specification order)
        # ------------------------------------------------------------------
        done = False

        # Step 1 ─ rotate the gripper to 90 deg about Z so it can enter the drawer side‑pose
        if not done:
            target_quat_ninety = R.from_euler("z", 90, degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat_ninety)
        # Step 2 ─ move to the bottom drawer's side pose
        if not done:
            obs, reward, done = move(env, task, bottom_side_pos)
        # Step 3 ─ move from the side pose to the anchor pose (handle)
        if not done:
            obs, reward, done = move(env, task, bottom_anchor_pos)
        # Step 4 ─ grasp the drawer handle (pick‑drawer equivalent)
        if not done:
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.05)
        # Step 5 ─ pull the drawer out (open)
        if not done:
            obs, reward, done = pull(env, task, pull_distance=0.10, pull_axis="x")
        # Step 6 ─ move away from the drawer (back to a neutral waypoint so items can be reached)
        if not done:
            neutral_pos = bottom_side_pos + np.array([0.0, -0.15, 0.10])  # slight offset
            obs, reward, done = move(env, task, neutral_pos)

        # -----------------------------
        #  HANDLE THE TWO TOMATOES
        # -----------------------------
        # Step 7 ─ pick the first tomato
        if not done:
            obs, reward, done = pick(env, task, tomato1_pos)
        # Step 8 ─ move to the plate
        if not done:
            obs, reward, done = move(env, task, plate_pos + np.array([0.0, 0.0, 0.10]))
        # Step 9 ─ place the first tomato
        if not done:
            obs, reward, done = place(env, task, plate_pos)
        # Step 10 ─ move back to the second tomato
        if not done:
            obs, reward, done = move(env, task, tomato2_pos + np.array([0.0, 0.0, 0.10]))
        # Step 11 ─ pick the second tomato
        if not done:
            obs, reward, done = pick(env, task, tomato2_pos)
        # Step 12 ─ move again over the plate
        if not done:
            obs, reward, done = move(env, task, plate_pos + np.array([0.0, 0.0, 0.10]))
        # Step 13 ─ place the second tomato
        if not done:
            obs, reward, done = place(env, task, plate_pos)

        # Final user‑friendly summary
        if done:
            print("[Task] RLBench task terminated early (done=True).")
        else:
            print("[Task] Plan executed. Drawer should be open and two tomatoes on the plate!")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
        raise  # re‑raise so any outer runner can see the failure
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
